﻿@extends('layouts.admin')

@push('styles')
<style>
    #voters-table td.editable { cursor: pointer; }
    #voters-table td.editable.is-editing { background: #fff8e1; }
    #voters-table td.editable.is-active { outline: 2px solid #0d6efd; outline-offset: -2px; }
    #voters-table td.editable.is-saving { opacity: 0.6; }
    #voters-table td.editable.is-error { outline: 2px solid #dc3545; outline-offset: -2px; }
    #voters-table td.editable input,
    #voters-table td.editable textarea,
    #voters-table td.editable select { width: 100%; font-size: 0.9rem; }
</style>
@endpush

@section('content')
<div class="d-flex justify-content-between align-items-center mb-3">
    <div>
        <h1 class="h5 mb-0">ভোটার তালিকা</h1>
        <p class="text-muted small mb-0">ফিল্টার করে অনুসন্ধান করুন</p>
    </div>
    <div class="d-flex flex-wrap gap-2 align-items-end">
        <a href="{{ route('admin.voters.create', array_filter(['source_file_id' => $filters['source_file_id'] ?? null], fn ($value) => $value !== null && $value !== '')) }}" class="btn btn-primary">নতুন ভোটার যোগ</a>
        <form method="GET" action="{{ route('admin.voters.print') }}" target="_blank" class="d-flex flex-wrap gap-2 align-items-end">
            @foreach(request()->query() as $key => $value)
                @if($key !== 'candidate_id' && !is_array($value))
                    <input type="hidden" name="{{ $key }}" value="{{ $value }}">
                @endif
            @endforeach
            <select name="candidate_id" class="form-select form-select-sm" {{ $candidates->isNotEmpty() ? 'required' : '' }}>
                <option value="">প্রার্থী নির্বাচন করুন</option>
                @foreach($candidates as $candidate)
                    <option value="{{ $candidate->id }}">{{ $candidate->name_bn }}</option>
                @endforeach
            </select>
            <button type="submit" class="btn btn-outline-secondary btn-sm">স্লিপ প্রিন্ট</button>
        </form>
        <div class="btn-group">
            <a href="{{ route('admin.voters.export.pdf', request()->query()) }}" class="btn btn-outline-secondary">PDF</a>
            <a href="{{ route('admin.voters.export.csv', request()->query()) }}" class="btn btn-outline-secondary">CSV</a>
        </div>
    </div>
</div>

<div class="card shadow-sm border-0 mb-3">
    <div class="card-body">
        <form method="GET" action="{{ route('admin.voters.index') }}">
            <div class="row g-2 align-items-end">
                <div class="col-md-3">
                    <label class="form-label">সার্চ (নাম/নং/ঠিকানা)</label>
                    <input type="text" name="q" value="{{ $filters['q'] ?? '' }}" class="form-control" placeholder="যেমন: শামীম বা 3306...">
                </div>
                <div class="col-md-2">
                    <label class="form-label">পেশা</label>
                    <select name="profession_bn" class="form-select">
                        <option value="">সব</option>
                        @foreach($professions as $profession)
                            <option value="{{ $profession }}" @selected(($filters['profession_bn'] ?? '') == $profession)>{{ $profession }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-1">
                    <label class="form-label">লিঙ্গ</label>
                    <select name="gender" class="form-select">
                        <option value="">সব</option>
                        <option value="male" @selected(($filters['gender'] ?? '') === 'male')>পুরুষ</option>
                        <option value="female" @selected(($filters['gender'] ?? '') === 'female')>মহিলা</option>
                        <option value="unknown" @selected(($filters['gender'] ?? '') === 'unknown')>অজানা</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">প্রতি পেজ</label>
                    <select name="per_page" class="form-select">
                        @foreach([300, 500, 1000] as $size)
                            <option value="{{ $size }}" @selected((int)($filters['per_page'] ?? 300) === $size)>{{ $size }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">সর্বনিম্ন বয়স</label>
                    <input type="number" name="min_age" value="{{ $filters['min_age'] ?? '' }}" class="form-control" placeholder="18">
                </div>
                <div class="col-md-2">
                    <label class="form-label">সর্বোচ্চ বয়স</label>
                    <input type="number" name="max_age" value="{{ $filters['max_age'] ?? '' }}" class="form-control" placeholder="60">
                </div>
                <div class="col-md-2">
                    <label class="form-label">জন্মতারিখ</label>
                    <input type="text" name="date_of_birth" value="{{ $filters['date_of_birth'] ?? '' }}" class="form-control" placeholder="উদাহরণ: ০১/০৭/১৯৮৫" inputmode="numeric">
                </div>
                <div class="col-md-2">
                    <label class="form-label">সোর্স ফাইল</label>
                    <select name="source_file_id" class="form-select">
                        <option value="">সব</option>
                        @foreach($sourceFiles as $file)
                            <option value="{{ $file->id }}" @selected(($filters['source_file_id'] ?? '') == $file->id)>
                                {{ $file->original_filename }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-3 text-end">
                    <button type="submit" class="btn btn-primary me-2">ফিল্টার করুন</button>
                    <a href="{{ route('admin.voters.index') }}" class="btn btn-outline-secondary">রিসেট</a>
                </div>
            </div>
        </form>
    </div>
</div>

<div class="card shadow-sm border-0">
    <div class="card-body">
        <div class="d-flex justify-content-between align-items-center mb-2">
            <div class="text-muted small">Showing {{ $voters->count() }} of {{ $voters->total() }} voters</div>
            <div class="text-muted small">Page {{ $voters->currentPage() }} / {{ $voters->lastPage() }}</div>
        </div>
        <div class="table-responsive">
            <table class="table align-middle" id="voters-table">
                <thead>
                <tr>
                    <th><input type="checkbox" disabled></th>
                    <th>ক্রমিক</th>
                    <th>নাম</th>
                    <th>ভোটার নং</th>
                    <th>পিতা</th>
                    <th>মাতা</th>
                    <th>পেশা</th>
                    <th>জন্মতারিখ</th>
                    <th>ঠিকানা</th>
                    <th>এলাকা</th>
                    <th></th>
                </tr>
                </thead>
                <tbody>
                @php
                    use Illuminate\Support\Str;
                    $bnDigits = ['0' => '০', '1' => '১', '2' => '২', '3' => '৩', '4' => '৪', '5' => '৫', '6' => '৬', '7' => '৭', '8' => '৮', '9' => '৯'];
                    $bnToEnDigits = ['০' => '0', '১' => '1', '২' => '2', '৩' => '3', '৪' => '4', '৫' => '5', '৬' => '6', '৭' => '7', '৮' => '8', '৯' => '9'];
                @endphp
                @forelse($voters as $voter)
                    <tr data-voter-id="{{ $voter->id }}">
                        <td><input type="checkbox"></td>
                        <td class="editable" tabindex="0" data-field="serial_no_pdf_bn" data-value="{{ $voter->serial_no_pdf_bn ?? $voter->serial_no }}">{{ $voter->serial_no_pdf_bn ?? $voter->serial_no }}</td>
                        <td class="editable fw-semibold" tabindex="0" data-field="name_bn" data-value="{{ $voter->name_bn }}">{{ $voter->name_bn }}</td>
                        <td class="editable" tabindex="0" data-field="voter_no" data-value="{{ $voter->voter_no }}">{{ $voter->voter_no }}</td>
                        <td class="editable" tabindex="0" data-field="father_name_bn" data-value="{{ $voter->father_name_bn }}">{{ Str::limit($voter->father_name_bn, 20) }}</td>
                        <td class="editable" tabindex="0" data-field="mother_name_bn" data-value="{{ $voter->mother_name_bn }}">{{ Str::limit($voter->mother_name_bn, 20) }}</td>
                        <td class="editable" tabindex="0" data-field="profession_bn" data-value="{{ $voter->profession_bn }}">{{ $voter->profession_bn }}</td>
                        @php
                            $dobBn = $voter->date_of_birth_bn ?? (optional($voter->date_of_birth)->format('d-m-Y') ? strtr(optional($voter->date_of_birth)->format('d-m-Y'), $bnDigits) : null);
                            $serialNoEn = $voter->serial_no ?? null;
                            if ($serialNoEn === null && $voter->serial_no_pdf_bn) {
                                $serialNoEn = (int) strtr($voter->serial_no_pdf_bn, $bnToEnDigits);
                            }
                            $nextSerialBn = null;
                            if ($serialNoEn !== null) {
                                $nextSerialEn = str_pad((string) ($serialNoEn + 1), 3, '0', STR_PAD_LEFT);
                                $nextSerialBn = strtr($nextSerialEn, $bnDigits);
                            }
                        @endphp
                        <td class="editable" tabindex="0" data-field="date_of_birth_bn" data-value="{{ $dobBn }}">{{ $dobBn }}</td>
                        <td class="editable" tabindex="0" data-field="address_bn" data-type="textarea" data-value="{{ $voter->address_bn }}">{{ Str::limit($voter->address_bn, 30) }}</td>
                        <td class="editable" tabindex="0" data-field="area_id" data-type="select" data-value="{{ $voter->area_id }}" data-display="{{ $voter->area->area_name_bn ?? '' }}">{{ $voter->area->area_name_bn ?? '' }}</td>
                        <td class="text-end">
                            @if($nextSerialBn && $voter->source_file_id)
                                <a href="{{ route('admin.voters.create', array_filter(['source_file_id' => $voter->source_file_id, 'serial_no_pdf_bn' => $nextSerialBn], fn ($value) => $value !== null && $value !== '')) }}" class="btn btn-sm btn-outline-success">এখানে যোগ</a>
                            @endif
                            <a href="{{ route('admin.voters.show', $voter) }}" class="btn btn-sm btn-outline-primary">দেখুন</a>
                            <a href="{{ route('admin.voters.edit', $voter) }}" class="btn btn-sm btn-outline-secondary">সম্পাদনা</a>
                        </td>
                    </tr>
                @empty
                    <tr>
                        <td colspan="11" class="text-center text-muted">কোনো ভোটার পাওয়া যায়নি</td>
                    </tr>
                @endforelse
                </tbody>
            </table>
        </div>
        <div class="mt-2">
            {{ $voters->withQueryString()->links('pagination::bootstrap-5') }}
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
(() => {
    const table = document.getElementById('voters-table');
    if (!table) return;

    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';
    const areas = @json($areas->map(fn($area) => ['id' => $area->id, 'name' => $area->area_name_bn]));
    const inlineBase = @json(url('/admin/voters'));

    const limitText = (value, length) => {
        if (!value) return '';
        return value.length > length ? `${value.slice(0, length - 1)}…` : value;
    };

    const formatDisplay = (field, value) => {
        if (field === 'father_name_bn' || field === 'mother_name_bn') {
            return limitText(value, 20);
        }
        if (field === 'address_bn') {
            return limitText(value, 30);
        }
        return value ?? '';
    };

    const buildUrl = (id) => `${inlineBase}/${id}/inline`;

    const getGrid = () => {
        const rows = Array.from(table.querySelectorAll('tbody tr'));
        return rows.map((row) => Array.from(row.querySelectorAll('td.editable')));
    };

    const findPosition = (cell, grid) => {
        for (let r = 0; r < grid.length; r++) {
            const idx = grid[r].indexOf(cell);
            if (idx !== -1) {
                return { row: r, col: idx };
            }
        }
        return null;
    };

    const moveFocus = (cell, direction) => {
        const grid = getGrid();
        const pos = findPosition(cell, grid);
        if (!pos) return null;
        let { row, col } = pos;

        const clampRow = (targetRow) => Math.max(0, Math.min(grid.length - 1, targetRow));
        const rowLength = (targetRow) => (grid[targetRow] ? grid[targetRow].length : 0);

        if (direction === 'right') {
            col++;
            if (rowLength(row) && col >= rowLength(row)) {
                if (row < grid.length - 1) {
                    row = row + 1;
                    col = 0;
                } else {
                    col = rowLength(row) - 1;
                }
            }
        } else if (direction === 'left') {
            col--;
            if (col < 0) {
                if (row > 0) {
                    row = row - 1;
                    col = rowLength(row) - 1;
                } else {
                    col = 0;
                }
            }
        } else if (direction === 'down') {
            row = clampRow(row + 1);
            if (col >= rowLength(row)) {
                col = rowLength(row) - 1;
            }
        } else if (direction === 'up') {
            row = clampRow(row - 1);
            if (col >= rowLength(row)) {
                col = rowLength(row) - 1;
            }
        }

        if (!grid[row] || !grid[row][col]) {
            return null;
        }

        return grid[row][col];
    };

    const setActiveCell = (cell) => {
        if (!cell) return;
        const prev = table.querySelector('td.editable.is-active');
        if (prev && prev !== cell) {
            prev.classList.remove('is-active');
        }
        cell.classList.add('is-active');
        const active = document.activeElement;
        if (cell.contains(active) && active && active.matches('input, textarea, select')) {
            cell.scrollIntoView({ block: 'nearest', inline: 'nearest' });
            return;
        }
        cell.focus({ preventScroll: true });
        cell.scrollIntoView({ block: 'nearest', inline: 'nearest' });
    };

    const cancelEdit = (cell) => {
        cell.classList.remove('is-editing');
        cell.classList.remove('is-saving');
        cell.classList.remove('is-error');
        const original = cell.dataset.original ?? '';
        const display = cell.dataset.display ?? original;
        cell.textContent = formatDisplay(cell.dataset.field, display);
        cell.dataset.value = original;
        setActiveCell(cell);
    };

    const saveCell = async (cell, input) => {
        const row = cell.closest('tr');
        const voterId = row?.dataset.voterId;
        if (!voterId) {
            cancelEdit(cell);
            return false;
        }

        const field = cell.dataset.field;
        const original = cell.dataset.original ?? '';
        const value = input.value ?? '';
        if (String(value).trim() === String(original).trim()) {
            cancelEdit(cell);
            return true;
        }

        cell.classList.remove('is-error');
        cell.classList.add('is-saving');

        try {
            const response = await fetch(buildUrl(voterId), {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': csrfToken,
                },
                body: JSON.stringify({ field, value }),
            });

            if (!response.ok) {
                const error = await response.json().catch(() => ({}));
                throw new Error(error.message || 'আপডেট ব্যর্থ হয়েছে');
            }

            const data = await response.json();
            const display = data.display ?? value;
            cell.classList.remove('is-editing');
            cell.classList.remove('is-saving');
            cell.textContent = formatDisplay(field, display);
            cell.dataset.value = data.value ?? display;
            if (field === 'area_id') {
                cell.dataset.display = display;
            }
            setActiveCell(cell);
            return true;
        } catch (error) {
            cell.classList.remove('is-saving');
            cell.classList.remove('is-editing');
            cell.classList.add('is-error');
            const display = cell.dataset.display ?? original;
            cell.textContent = formatDisplay(field, display);
            cell.dataset.value = original;
            cell.title = error.message;
            setActiveCell(cell);
            return false;
        }
    };

    const startEdit = (cell) => {
        if (cell.classList.contains('is-editing')) return;
        const field = cell.dataset.field;
        const type = cell.dataset.type || 'text';
        const current = cell.dataset.value ?? cell.textContent.trim();
        cell.dataset.original = current;
        setActiveCell(cell);
        cell.classList.add('is-editing');
        cell.classList.remove('is-error');
        cell.textContent = '';

        let input;
        if (type === 'textarea') {
            input = document.createElement('textarea');
            input.rows = 2;
            input.className = 'form-control form-control-sm';
        } else if (type === 'select') {
            input = document.createElement('select');
            input.className = 'form-select form-select-sm';
            areas.forEach((area) => {
                const option = document.createElement('option');
                option.value = String(area.id);
                option.textContent = area.name;
                input.appendChild(option);
            });
        } else {
            input = document.createElement('input');
            input.type = 'text';
            input.className = 'form-control form-control-sm';
        }

        input.value = current ?? '';
        cell.appendChild(input);
        input.focus();
        if (input.select) {
            input.select();
        }

        const commitWithNav = (direction) => {
            input.dataset.navDir = direction;
            input.blur();
        };

        input.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                event.preventDefault();
                cancelEdit(cell);
            }
            if (event.key === 'Tab') {
                event.preventDefault();
                commitWithNav(event.shiftKey ? 'left' : 'right');
            }
            if (event.key === 'Enter' && (type !== 'textarea' || !event.shiftKey)) {
                event.preventDefault();
                commitWithNav('down');
            }
        });

        input.addEventListener('blur', async () => {
            const navDir = input.dataset.navDir || '';
            delete input.dataset.navDir;
            const ok = await saveCell(cell, input);
            if (ok && navDir) {
                const next = moveFocus(cell, navDir);
                if (next) {
                    setActiveCell(next);
                }
            }
        });
    };

    table.addEventListener('click', (event) => {
        const cell = event.target.closest('td.editable');
        if (!cell || cell.querySelector('input, textarea, select')) return;
        startEdit(cell);
    });

    table.addEventListener('focusin', (event) => {
        if (event.target.matches('input, textarea, select')) {
            return;
        }
        const cell = event.target.closest('td.editable');
        if (cell) {
            setActiveCell(cell);
        }
    });

    table.addEventListener('keydown', (event) => {
        if (event.target.matches('input, textarea, select')) {
            return;
        }
        const cell = event.target.closest('td.editable');
        if (!cell) return;

        const keyMap = {
            ArrowRight: 'right',
            ArrowLeft: 'left',
            ArrowDown: 'down',
            ArrowUp: 'up',
        };
        if (event.key === 'Enter') {
            event.preventDefault();
            startEdit(cell);
            return;
        }
        if (event.key === 'Tab') {
            event.preventDefault();
            const next = moveFocus(cell, event.shiftKey ? 'left' : 'right');
            if (next) setActiveCell(next);
            return;
        }
        const direction = keyMap[event.key];
        if (!direction) return;
        event.preventDefault();
        const next = moveFocus(cell, direction);
        if (next) setActiveCell(next);
    });
})();
</script>
@endpush


