﻿@extends('layouts.admin')

@section('content')
@php
    $currentStatus = $sourceFile->import_status;
    if ($currentStatus === 'pending' && $sourceFile->parsed_successfully) {
        $currentStatus = 'completed';
    }
    $isRunning = in_array($currentStatus, ['queued', 'running'], true);
    $isCompleted = $currentStatus === 'completed';
    $isStopped = $currentStatus === 'stopped';
    $isRestartable = in_array($currentStatus, ['stopped', 'failed', 'completed'], true);
    $totalPages = $sourceFile->total_pages ?? 0;
    $importedPages = $sourceFile->imported_pages ?? 0;
    $percentage = $totalPages > 0 ? (int) round(($importedPages / $totalPages) * 100) : 0;
@endphp

<style>
    .scan-panel {
        border: 1px dashed #8ca0ad;
        background: linear-gradient(135deg, #f5f7f6 0%, #eef4f8 100%);
        border-radius: 14px;
        padding: 18px;
        box-shadow: inset 0 1px 0 rgba(255, 255, 255, 0.7);
    }
    .scan-paper {
        position: relative;
        height: 120px;
        border-radius: 12px;
        background: linear-gradient(180deg, #ffffff 0%, #fbfbf7 55%, #f0f6f1 100%);
        border: 1px solid #dbe3e8;
        overflow: hidden;
        box-shadow: 0 10px 20px rgba(15, 23, 42, 0.12);
        animation: paperFloat 3.2s ease-in-out infinite;
    }
    .scan-paper::before {
        content: "";
        position: absolute;
        inset: -30% 0 0;
        background: radial-gradient(ellipse at center, rgba(64, 200, 120, 0.18) 0%, rgba(64, 200, 120, 0) 70%);
        transform: translateY(-60%);
        animation: beamSweep 2.1s linear infinite;
        pointer-events: none;
    }
    .scan-paper::after {
        content: "";
        position: absolute;
        inset: 0;
        background-image:
            linear-gradient(90deg, rgba(15, 23, 42, 0.06) 1px, transparent 1px),
            linear-gradient(180deg, rgba(15, 23, 42, 0.04) 1px, transparent 1px);
        background-size: 18px 100%, 100% 18px;
        background-position: 0 0, 0 0;
        opacity: 0.28;
        animation: gridDrift 2.8s linear infinite;
        pointer-events: none;
    }
    .scan-line {
        position: absolute;
        left: -15%;
        right: -15%;
        height: 8px;
        background: linear-gradient(90deg, transparent, rgba(61, 198, 120, 0.95), transparent);
        animation: scanMove 1.8s linear infinite;
        box-shadow: 0 0 12px rgba(59, 178, 115, 0.45), 0 0 24px rgba(59, 178, 115, 0.2);
        filter: blur(0.2px);
    }
    .scan-label {
        position: absolute;
        bottom: 12px;
        left: 16px;
        font-size: 12px;
        color: #0f172a;
        background: rgba(255, 255, 255, 0.9);
        padding: 6px 12px;
        border-radius: 999px;
        border: 1px solid #d8e2ea;
        box-shadow: 0 6px 12px rgba(15, 23, 42, 0.12);
        letter-spacing: 0.2px;
        display: inline-flex;
        align-items: center;
        gap: 6px;
        overflow: hidden;
    }
    .scan-text {
        position: relative;
        display: inline-block;
        white-space: nowrap;
        animation: textDrift 2.6s ease-in-out infinite;
    }
    .scan-text::after {
        content: "";
        position: absolute;
        inset: 0;
        background: linear-gradient(120deg, transparent 0%, rgba(61, 198, 120, 0.35) 45%, transparent 70%);
        transform: translateX(-120%);
        animation: textSweep 2.6s ease-in-out infinite;
        pointer-events: none;
    }
    .scan-dots span {
        display: inline-block;
        width: 5px;
        height: 5px;
        margin-left: 4px;
        border-radius: 999px;
        background: #3bb273;
        box-shadow: 0 0 6px rgba(59, 178, 115, 0.6);
        animation: scanBlink 1.2s infinite ease-in-out;
    }
    .scan-dots span:nth-child(2) { animation-delay: 0.2s; }
    .scan-dots span:nth-child(3) { animation-delay: 0.4s; }

    @keyframes scanMove {
        0% { top: -12%; opacity: 0.4; }
        50% { opacity: 1; }
        100% { top: 112%; opacity: 0.4; }
    }
    @keyframes scanBlink {
        0%, 100% { opacity: 0.3; transform: translateY(0); }
        50% { opacity: 1; transform: translateY(-3px); }
    }
    @keyframes textDrift {
        0%, 100% { transform: translateX(0); }
        50% { transform: translateX(6px); }
    }
    @keyframes textSweep {
        0% { transform: translateX(-120%); opacity: 0; }
        20% { opacity: 0.7; }
        100% { transform: translateX(120%); opacity: 0; }
    }
    @keyframes paperFloat {
        0%, 100% { transform: translateY(0); }
        50% { transform: translateY(-2px); }
    }
    @keyframes beamSweep {
        0% { transform: translateY(-60%); opacity: 0; }
        20% { opacity: 0.6; }
        60% { opacity: 0.35; }
        100% { transform: translateY(120%); opacity: 0; }
    }
    @keyframes gridDrift {
        0% { background-position: 0 0, 0 0; }
        100% { background-position: 36px 0, 0 36px; }
    }
</style>

<div class="d-flex justify-content-between align-items-center mb-3">
    <div>
        <h1 class="h5 mb-0">পিডিএফ ইম্পোর্ট</h1>
        <p class="text-muted small mb-0">ফাইল: {{ $sourceFile->original_filename }}</p>
    </div>
    <a href="{{ route('admin.source-files.index') }}" class="btn btn-outline-secondary">ফাইল তালিকা</a>
</div>

<div class="card shadow-sm border-0 mb-3">
    <div class="card-body">
        <div class="row g-3 mb-3">
            <div class="col-md-4">
                <div class="text-muted small mb-1">এলাকা</div>
                <div class="fw-semibold">{{ $sourceFile->area->area_name_bn ?? '-' }}</div>
            </div>
            <div class="col-md-4">
                <div class="text-muted small mb-1">পেজ সংখ্যা</div>
                <div class="fw-semibold">{{ $pageCount ? $pageCount : 'অজানা' }}</div>
            </div>
            <div class="col-md-4">
                <div class="text-muted small mb-1">স্ট্যাটাস</div>
                @switch($currentStatus)
                    @case('completed')
                        <span class="badge bg-success" id="import-status-badge">সম্পন্ন</span>
                        @break
                    @case('failed')
                        <span class="badge bg-danger" id="import-status-badge">ব্যর্থ</span>
                        @break
                    @case('stopped')
                        <span class="badge bg-secondary" id="import-status-badge">বন্ধ</span>
                        @break
                    @case('running')
                    @case('queued')
                        <span class="badge bg-warning text-dark" id="import-status-badge">চলমান</span>
                        @break
                    @default
                        <span class="badge bg-secondary" id="import-status-badge">পেন্ডিং</span>
                @endswitch
            </div>
        </div>

        <form action="{{ route('admin.source-files.import.start', $sourceFile) }}" method="POST">
            @csrf
            <div class="row g-3 align-items-end">
                <div class="col-md-3">
                    <label class="form-label">ইম্পোর্ট ধরন</label>
                    <select name="page_mode" id="page_mode" class="form-select" @disabled($isRunning)>
                        <option value="all" @selected(old('page_mode', 'all') === 'all')>সব পেজ</option>
                        <option value="single" @selected(old('page_mode') === 'single')>একটি পেজ</option>
                        <option value="range" @selected(old('page_mode') === 'range')>পেজ রেঞ্জ</option>
                    </select>
                    @error('page_mode')
                        <div class="text-danger small">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-3" data-field="single">
                    <label class="form-label">পেজ নম্বর</label>
                    <input type="number" name="page_number" value="{{ old('page_number') }}" min="1" class="form-control" @disabled($isRunning)>
                    @error('page_number')
                        <div class="text-danger small">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-3" data-field="range">
                    <label class="form-label">শুরু পেজ</label>
                    <input type="number" name="page_from" value="{{ old('page_from') }}" min="1" class="form-control" @disabled($isRunning)>
                    @error('page_from')
                        <div class="text-danger small">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-3" data-field="range">
                    <label class="form-label">শেষ পেজ</label>
                    <input type="number" name="page_to" value="{{ old('page_to') }}" min="1" class="form-control" @disabled($isRunning)>
                    @error('page_to')
                        <div class="text-danger small">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-3">
                    <label class="form-label">Voter Gender</label>
                    <select name="voter_gender" class="form-select" @disabled($isRunning)>
                        <option value="">-- Select --</option>
                        <option value="male" @selected(old('voter_gender', $importOptions['default_voter_gender'] ?? $importOptions['voter_gender'] ?? null) === 'male')>Male</option>
                        <option value="female" @selected(old('voter_gender', $importOptions['default_voter_gender'] ?? $importOptions['voter_gender'] ?? null) === 'female')>Female</option>
                        <option value="unknown" @selected(old('voter_gender', $importOptions['default_voter_gender'] ?? $importOptions['voter_gender'] ?? null) === 'unknown')>Unknown</option>
                    </select>
                    @error('voter_gender')
                        <div class="text-danger small">{{ $message }}</div>
                    @enderror
                </div>
                <div class="col-md-3">
                    <button type="submit" class="btn btn-primary w-100" @disabled($isRunning)>
                        ইম্পোর্ট শুরু
                    </button>
                </div>
            </div>
            <div class="form-text mt-2">ইম্পোর্ট শুরু হলে OCR চালু হবে এবং পেজভিত্তিক ডাটা ডাটাবেজে যাবে।</div>
        </form>
    </div>
</div>

@if($currentStatus !== 'pending')
    <div class="card shadow-sm border-0">
        <div class="card-body">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <div class="fw-semibold">ইম্পোর্ট প্রোগ্রেস</div>
                <div class="text-muted small" id="progress-text">{{ $importedPages }} / {{ $totalPages }}</div>
            </div>
            <div class="progress" id="import-progress"
                 data-url="{{ route('admin.source-files.import.status', $sourceFile) }}"
                 data-status="{{ $currentStatus }}"
                 data-voters-url="{{ route('admin.voters.index', ['source_file_id' => $sourceFile->id]) }}"
                 data-imported="{{ $importedPages }}"
                 data-total="{{ $totalPages }}">
                <div class="progress-bar" role="progressbar" style="width: {{ $percentage }}%">
                    <span id="progress-percentage">{{ $percentage }}%</span>
                </div>
            </div>
            <div id="import-auth-warning" class="alert alert-warning mt-3 d-none">
                সেশন শেষ হয়েছে। অনুগ্রহ করে পেজ রিফ্রেশ করে আবার লগইন করুন।
            </div>
            <div class="d-flex flex-wrap gap-2 justify-content-end mt-3">
                @if($isRunning)
                    <form method="POST" action="{{ route('admin.source-files.import.stop', $sourceFile) }}" onsubmit="return confirm('ইম্পোর্ট বন্ধ করবেন?')">
                        @csrf
                        <button type="submit" class="btn btn-outline-danger btn-sm">ইম্পোর্ট বন্ধ</button>
                    </form>
                @elseif($isRestartable)
                    <form method="POST" action="{{ route('admin.source-files.import.restart', $sourceFile) }}" onsubmit="return confirm('ইম্পোর্ট আবার শুরু করবেন?')">
                        @csrf
                        <button type="submit" class="btn btn-outline-primary btn-sm">ইম্পোর্ট রিস্টার্ট</button>
                    </form>
                @endif
            </div>

            <div id="scan-animation" class="scan-panel mt-3 {{ $isRunning ? '' : 'd-none' }}" aria-live="polite">
                <div class="scan-paper">
                    <div class="scan-line"></div>
                    <div class="scan-label">
                        <span class="scan-text">PDF স্ক্যান করা হচ্ছে</span>
                        <span class="scan-dots"><span></span><span></span><span></span></span>
                    </div>
                </div>
            </div>

            <div id="import-success" class="alert alert-success d-flex align-items-center justify-content-between mt-3 {{ $isCompleted ? '' : 'd-none' }}">
                <div>ইম্পোর্ট সম্পন্ন হয়েছে।</div>
                <a href="{{ route('admin.voters.index', ['source_file_id' => $sourceFile->id]) }}" class="btn btn-success btn-sm">
                    ভোটার তালিকা দেখুন
                </a>
            </div>
        </div>
    </div>
@endif

<script>
    (function () {
        const pageMode = document.getElementById('page_mode');
        const rangeFields = document.querySelectorAll('[data-field="range"]');
        const singleFields = document.querySelectorAll('[data-field="single"]');

        function toggleFields() {
            const mode = pageMode ? pageMode.value : 'all';
            rangeFields.forEach(field => field.style.display = mode === 'range' ? '' : 'none');
            singleFields.forEach(field => field.style.display = mode === 'single' ? '' : 'none');
        }

        if (pageMode) {
            pageMode.addEventListener('change', toggleFields);
            toggleFields();
        }

        const progress = document.getElementById('import-progress');
        if (!progress) {
            return;
        }

        const url = progress.dataset.url;
        const progressText = document.getElementById('progress-text');
        const progressPercentage = document.getElementById('progress-percentage');
        const scanPanel = document.getElementById('scan-animation');
        const successBanner = document.getElementById('import-success');
        const authWarning = document.getElementById('import-auth-warning');
        const statusBadge = document.getElementById('import-status-badge');
        let completedShown = progress.dataset.status === 'completed';

        const render = (data) => {
            const total = Number(data.total_pages || 0);
            const imported = Number(data.imported_pages || 0);
            const percentage = total > 0 ? Math.round((imported / total) * 100) : 0;
            progress.querySelector('.progress-bar').style.width = percentage + '%';
            if (progressText) {
                progressText.textContent = imported + ' / ' + total;
            }
            if (progressPercentage) {
                progressPercentage.textContent = percentage + '%';
            }
        };

        const updateStatusUI = (status) => {
            const running = status === 'running' || status === 'queued';
            if (scanPanel) {
                scanPanel.classList.toggle('d-none', !running);
            }
            if (successBanner && status === 'completed' && !completedShown) {
                successBanner.classList.remove('d-none');
                completedShown = true;
            } else if (successBanner && status !== 'completed' && !completedShown) {
                successBanner.classList.add('d-none');
            }
            if (statusBadge) {
                if (status === 'completed') {
                    statusBadge.className = 'badge bg-success';
                    statusBadge.textContent = 'সম্পন্ন';
                } else if (status === 'failed') {
                    statusBadge.className = 'badge bg-danger';
                    statusBadge.textContent = 'ব্যর্থ';
                } else if (status === 'stopped') {
                    statusBadge.className = 'badge bg-secondary';
                    statusBadge.textContent = 'বন্ধ';
                } else if (running) {
                    statusBadge.className = 'badge bg-warning text-dark';
                    statusBadge.textContent = 'চলমান';
                } else {
                    statusBadge.className = 'badge bg-secondary';
                    statusBadge.textContent = 'পেন্ডিং';
                }
            }
        };

        const showAuthWarning = () => {
            if (authWarning) {
                authWarning.classList.remove('d-none');
            }
            if (scanPanel) {
                scanPanel.classList.add('d-none');
            }
            if (statusBadge) {
                statusBadge.className = 'badge bg-danger';
                statusBadge.textContent = 'সেশন শেষ';
            }
        };

        const poll = async () => {
            try {
                const response = await fetch(url, { headers: { 'Accept': 'application/json' } });
                const contentType = response.headers.get('content-type') || '';
                const redirectedToLogin = response.redirected && response.url.includes('/login');
                if (redirectedToLogin || !contentType.includes('application/json')) {
                    showAuthWarning();
                    return;
                }
                if (!response.ok) {
                    return;
                }
                const data = await response.json();
                render(data);
                updateStatusUI(data.status);
                if (data.status === 'running' || data.status === 'queued') {
                    setTimeout(poll, 3000);
                }
            } catch (error) {
                // ignore
            }
        };

        render({
            imported_pages: progress.dataset.imported,
            total_pages: progress.dataset.total
        });
        updateStatusUI(progress.dataset.status);
        if (progress.dataset.status === 'running' || progress.dataset.status === 'queued') {
            poll();
        }
    })();
</script>
@endsection

