<?php

namespace App\Services;

use App\Models\SourceFile;
use App\Models\Voter;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;

class VoterCsvImportService
{
    private const EXPECTED_HEADERS = [
        'serial_no_pdf_bn',
        'voter_no',
        'name_bn',
        'father_name_bn',
        'mother_name_bn',
        'profession_bn',
        'date_of_birth_bn',
        'address_bn',
    ];

    public function import(SourceFile $sourceFile): array
    {
        $sourceFile->loadMissing('area');
        $filePath = Storage::path($sourceFile->stored_path);

        if (! file_exists($filePath)) {
            throw new \RuntimeException('CSV file not found: '.$filePath);
        }

        $summary = [
            'total_rows' => 0,
            'imported_rows' => 0,
            'failed_rows' => 0,
            'failed_samples' => [],
        ];

        $csv = $this->openCsv($filePath);
        $headerMap = null;

        foreach ($csv as $row) {
            if (! is_array($row)) {
                continue;
            }

            $row = $this->normalizeRow($row);
            if ($this->rowIsEmpty($row)) {
                continue;
            }

            if ($headerMap === null) {
                $headerRow = $this->normalizeHeaderRow($row);
                if ($this->looksLikeHeader($headerRow)) {
                    $headerMap = $this->buildHeaderMap($headerRow);
                    continue;
                }
                $headerMap = $this->defaultHeaderMap();
            }

            $summary['total_rows']++;
            $payload = $this->mapRow($row, $headerMap);
            $parsed = $this->normalizeRecord($payload, $sourceFile);

            if (empty($parsed['voter_no']) || empty($parsed['name_bn'])) {
                $summary['failed_rows']++;
                $this->pushFailedSample($summary['failed_samples'], $payload);
                continue;
            }

            $existing = Voter::where('area_id', $sourceFile->area_id)
                ->where('voter_no', $parsed['voter_no'])
                ->first();

            $data = array_merge($parsed, [
                'area_id' => $sourceFile->area_id,
                'source_file_id' => $sourceFile->id,
                'extra_json' => [
                    'raw_row' => $payload,
                ],
            ]);

            if ($existing) {
                $update = $this->buildMergeData($existing->toArray(), $data);
                if (! empty($update)) {
                    $existing->update($update);
                }
            } else {
                Voter::create($data);
            }

            $summary['imported_rows']++;
        }

        return $summary;
    }

    private function openCsv(string $filePath): \SplFileObject
    {
        $csv = new \SplFileObject($filePath);
        $csv->setFlags(\SplFileObject::READ_CSV | \SplFileObject::SKIP_EMPTY | \SplFileObject::READ_AHEAD);
        $csv->setCsvControl(',');

        return $csv;
    }

    private function normalizeRow(array $row): array
    {
        $normalized = [];
        foreach ($row as $index => $value) {
            if ($index === 0 && is_string($value)) {
                $value = $this->stripBom($value);
            }
            $normalized[] = is_string($value) ? trim($value) : $value;
        }

        return $normalized;
    }

    private function rowIsEmpty(array $row): bool
    {
        foreach ($row as $value) {
            if (is_string($value)) {
                if (trim($value) !== '') {
                    return false;
                }
                continue;
            }
            if ($value !== null && $value !== '') {
                return false;
            }
        }

        return true;
    }

    private function normalizeHeaderRow(array $row): array
    {
        $normalized = [];
        foreach ($row as $index => $value) {
            $value = is_string($value) ? $value : '';
            if ($index === 0) {
                $value = $this->stripBom($value);
            }
            $value = mb_strtolower(trim($value));
            $normalized[] = $value;
        }

        return $normalized;
    }

    private function looksLikeHeader(array $row): bool
    {
        $hits = 0;
        foreach (self::EXPECTED_HEADERS as $header) {
            if (in_array($header, $row, true)) {
                $hits++;
            }
        }

        return $hits >= 3;
    }

    private function buildHeaderMap(array $headerRow): array
    {
        $map = [];
        foreach ($headerRow as $index => $name) {
            if (in_array($name, self::EXPECTED_HEADERS, true)) {
                $map[$name] = $index;
            }
        }

        return $map;
    }

    private function defaultHeaderMap(): array
    {
        $map = [];
        foreach (self::EXPECTED_HEADERS as $index => $name) {
            $map[$name] = $index;
        }

        return $map;
    }

    private function mapRow(array $row, array $headerMap): array
    {
        $data = [];
        foreach (self::EXPECTED_HEADERS as $header) {
            $index = $headerMap[$header] ?? null;
            $data[$header] = $index !== null && array_key_exists($index, $row)
                ? $row[$index]
                : null;
        }

        return $data;
    }

    private function normalizeRecord(array $row, SourceFile $sourceFile): array
    {
        $serialBn = $this->keepBanglaDigits($row['serial_no_pdf_bn'] ?? null);
        $serialNo = $serialBn ? (int) $this->bnToEnDigits($serialBn) : null;
        if ($serialNo === 0) {
            $serialNo = null;
        }

        $voterNo = $this->keepBanglaDigits($row['voter_no'] ?? null);
        if (! $this->isValidVoterNo($voterNo)) {
            $voterNo = null;
        }

        $dobBn = $this->normalizeDateBn($row['date_of_birth_bn'] ?? null);
        $dob = $this->parseDate($dobBn);

        return [
            'serial_no' => $serialNo,
            'serial_no_pdf_bn' => $serialBn,
            'voter_no' => $voterNo,
            'name_bn' => $this->normalizeText($row['name_bn'] ?? null),
            'father_name_bn' => $this->normalizeText($row['father_name_bn'] ?? null),
            'mother_name_bn' => $this->normalizeText($row['mother_name_bn'] ?? null),
            'profession_bn' => $this->normalizeText($row['profession_bn'] ?? null),
            'date_of_birth' => $dob,
            'date_of_birth_bn' => $dobBn,
            'address_bn' => $this->normalizeText($row['address_bn'] ?? null),
            'name_en' => null,
            'gender' => $this->mapGenderFromArea($sourceFile),
        ];
    }

    private function buildMergeData(array $existing, array $incoming): array
    {
        $update = [];
        foreach ([
            'serial_no',
            'serial_no_pdf_bn',
            'name_bn',
            'father_name_bn',
            'mother_name_bn',
            'profession_bn',
            'date_of_birth',
            'date_of_birth_bn',
            'address_bn',
        ] as $field) {
            if ($this->isBlankValue($existing[$field] ?? null) && ! $this->isBlankValue($incoming[$field] ?? null)) {
                $update[$field] = $incoming[$field];
            }
        }

        if (empty($existing['source_file_id']) && ! empty($incoming['source_file_id'])) {
            $update['source_file_id'] = $incoming['source_file_id'];
        }

        return $update;
    }

    private function normalizeText(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }
        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $value = preg_replace('/\s+/u', ' ', $value);

        return $value === '' ? null : $value;
    }

    private function normalizeDateBn(?string $value): ?string
    {
        $value = $this->normalizeText($value);
        if (! $value) {
            return null;
        }

        $value = $this->normalizeBnDigits($value);
        $value = str_replace('-', '/', $value);
        $value = preg_replace('/[^০-৯\/]/u', '', $value);

        return $value === '' ? null : $value;
    }

    private function keepBanglaDigits(?string $value): ?string
    {
        $value = $this->normalizeText($value);
        if (! $value) {
            return null;
        }

        $value = $this->normalizeBnDigits($value);
        $value = preg_replace('/[^০-৯]/u', '', $value);

        return $value === '' ? null : $value;
    }

    private function bnToEnDigits(string $value): string
    {
        return strtr($value, [
            '০' => '0', '১' => '1', '২' => '2', '৩' => '3', '৪' => '4',
            '৫' => '5', '৬' => '6', '৭' => '7', '৮' => '8', '৯' => '9',
        ]);
    }

    private function normalizeBnDigits(string $value): string
    {
        return strtr($value, [
            '0' => '০', '1' => '১', '2' => '২', '3' => '৩', '4' => '৪',
            '5' => '৫', '6' => '৬', '7' => '৭', '8' => '৮', '9' => '৯',
        ]);
    }

    private function parseDate(?string $dateString): ?string
    {
        if (! $dateString) {
            return null;
        }

        $converted = $this->bnToEnDigits(trim($dateString));
        foreach (['d/m/Y', 'd-m-Y', 'Y-m-d'] as $format) {
            try {
                return Carbon::createFromFormat($format, $converted)->format('Y-m-d');
            } catch (\Throwable $e) {
                continue;
            }
        }

        return null;
    }

    private function isValidVoterNo(?string $value): bool
    {
        if (! $value) {
            return false;
        }

        $digits = $this->normalizeBnDigits($value);
        $digits = preg_replace('/[^০-৯]/u', '', $digits);
        $length = mb_strlen($digits);

        return $length >= 10 && $length <= 17;
    }

    private function mapGenderFromArea(SourceFile $sourceFile): string
    {
        return match ($sourceFile->area->gender_type ?? 'mixed') {
            'male' => 'male',
            'female' => 'female',
            default => 'unknown',
        };
    }

    private function isBlankValue(?string $value): bool
    {
        return $value === null || trim((string) $value) === '';
    }

    private function stripBom(string $value): string
    {
        return preg_replace('/^\xEF\xBB\xBF/', '', $value);
    }

    private function pushFailedSample(array &$samples, array $payload): void
    {
        if (count($samples) < 15) {
            $samples[] = $payload;
        }
    }
}
