<?php

namespace App\Services;

use App\Models\SourceFile;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Symfony\Component\Process\Process;

class PdfSlipPrintService
{
    public function build(SourceFile $sourceFile, int $pageFrom, int $pageTo, array $options = []): array
    {
        $filePath = $this->resolveStoredPath($sourceFile->stored_path);
        if (! $filePath || ! file_exists($filePath)) {
            throw new \RuntimeException('PDF file not found.');
        }

        $options = $this->normalizeOptions($options);
        $pages = range($pageFrom, $pageTo);

        $cacheKey = sha1($filePath.'|'.filemtime($filePath).'|'.filesize($filePath).'|'.json_encode($options));
        $token = 'pdfslip-'.substr($cacheKey, 0, 16);
        $baseDir = storage_path('app/pdf_slips/'.$token);
        File::ensureDirectoryExists($baseDir);

        $pageImages = [];
        foreach ($pages as $page) {
            $pageImages[$page] = $this->renderPageBlocks($filePath, $page, $options, $baseDir);
        }

        return [
            'token' => $token,
            'pages' => $pageImages,
            'options' => $options,
        ];
    }

    public function startCropSession(SourceFile $sourceFile, int $pageFrom, int $pageTo, array $options = []): array
    {
        $filePath = $this->resolveStoredPath($sourceFile->stored_path);
        if (! $filePath || ! file_exists($filePath)) {
            throw new \RuntimeException('PDF file not found.');
        }

        $options = $this->normalizeOptions($options);
        $pageFrom = max(1, $pageFrom);
        $pageTo = max($pageFrom, $pageTo);
        $rowsOpt = (int) ($options['rows'] ?? 0);
        $perPage = $rowsOpt > 0 ? ($options['cols'] * $rowsOpt) : 0;

        $cacheKey = sha1($filePath.'|'.filemtime($filePath).'|'.filesize($filePath).'|crop|'.$pageFrom.'|'.$pageTo.'|'.json_encode($options));
        $token = 'pdfslip-crop-'.substr($cacheKey, 0, 16);
        $baseDir = $this->cropBaseDir($sourceFile, $token);
        File::ensureDirectoryExists($baseDir);

        $statusPath = $baseDir.'/status.json';
        $status = $this->readJson($statusPath, []);
        $optionsHash = sha1(json_encode($options));
        if (! empty($status)) {
            $sameOptions = ($status['options_hash'] ?? '') === $optionsHash;
            $sameRange = ((int) ($status['page_from'] ?? 0) === $pageFrom) && ((int) ($status['page_to'] ?? 0) === $pageTo);
            if (! $sameOptions || ! $sameRange) {
                File::deleteDirectory($baseDir);
                File::ensureDirectoryExists($baseDir);
                $status = [];
            }
        }
        if (empty($status)) {
            $status = [
                'token' => $token,
                'source_file_id' => $sourceFile->id,
                'page_from' => $pageFrom,
                'page_to' => $pageTo,
                'total_pages' => ($pageTo - $pageFrom + 1),
                'per_page' => $perPage,
                'expected_blocks' => ($pageTo - $pageFrom + 1) * $perPage,
                'total_blocks' => 0,
                'done_pages' => 0,
                'done_blocks' => 0,
                'pages_done' => [],
                'page_counts' => [],
                'options' => $options,
                'options_hash' => $optionsHash,
                'started_at' => now()->toDateTimeString(),
                'updated_at' => now()->toDateTimeString(),
                'completed_at' => null,
            ];
            $this->writeJson($statusPath, $status);
        } else {
            $manifest = $this->readJson($baseDir.'/manifest.json', []);
            $pagesDone = $status['pages_done'] ?? [];
            $status['done_pages'] = count($pagesDone);
            $status['done_blocks'] = is_array($manifest) ? count($manifest) : 0;
            $status['updated_at'] = now()->toDateTimeString();
            $this->writeJson($statusPath, $status);
        }

        return $status;
    }

    public function cropPage(SourceFile $sourceFile, string $token, int $page): array
    {
        $baseDir = $this->cropBaseDir($sourceFile, $token);
        $statusPath = $baseDir.'/status.json';
        $status = $this->readJson($statusPath, []);
        if (empty($status)) {
            throw new \RuntimeException('Crop session not found.');
        }

        $pageFrom = (int) ($status['page_from'] ?? 1);
        $pageTo = (int) ($status['page_to'] ?? $pageFrom);
        if ($page < $pageFrom || $page > $pageTo) {
            throw new \RuntimeException('Invalid page number.');
        }

        $pagesDone = array_map('intval', $status['pages_done'] ?? []);
        if (in_array($page, $pagesDone, true)) {
            return $status;
        }

        $filePath = $this->resolveStoredPath($sourceFile->stored_path);
        if (! $filePath || ! file_exists($filePath)) {
            throw new \RuntimeException('PDF file not found.');
        }

        $options = $status['options'] ?? $this->normalizeOptions([]);
        $files = $this->renderPageBlocks($filePath, $page, $options, $baseDir);
        $files = $this->sortBlockFiles($files);
        $pageCount = count($files);

        $perPage = (int) ($status['per_page'] ?? 0);
        if ($perPage <= 0) {
            $rowsHint = (int) ($options['rows'] ?? 0);
            if ($rowsHint <= 0) {
                $rowsHint = $this->resolveRows($options);
            }
            $perPage = max(1, (int) ($options['cols'] ?? 1) * $rowsHint);
        }
        $manifestPath = $baseDir.'/manifest.json';
        $manifest = $this->readJson($manifestPath, []);
        if (! is_array($manifest)) {
            $manifest = [];
        }

        $existingSerials = [];
        foreach ($manifest as $entry) {
            if (isset($entry['serial'])) {
                $existingSerials[(int) $entry['serial']] = true;
            }
        }

        $pageCounts = $status['page_counts'] ?? [];
        $expectedCount = $this->inferExpectedBlockCount($pageCounts, $options);
        if ($expectedCount !== null && $pageCount > 0 && $pageCount !== $expectedCount) {
            $forcedRows = (int) round($expectedCount / max(1, (int) $options['cols']));
            if ($forcedRows > 0) {
                $forcedOptions = $options;
                $forcedOptions['rows'] = $forcedRows;
                $this->purgePageCache($baseDir, $page);
                $files = $this->renderPageBlocks($filePath, $page, $forcedOptions, $baseDir);
                $files = $this->sortBlockFiles($files);
                $pageCount = count($files);
            }
        }

        $startSerial = $this->computeStartSerial($page, $pageCounts, $pageFrom, (int) ($status['done_blocks'] ?? 0));
        foreach ($files as $index => $file) {
            $serial = $startSerial + $index;
            if (isset($existingSerials[$serial])) {
                continue;
            }
            $manifest[] = [
                'serial' => $serial,
                'page' => $page,
                'block' => $index,
                'file' => $file,
            ];
        }

        usort($manifest, fn ($a, $b) => ($a['serial'] ?? 0) <=> ($b['serial'] ?? 0));
        $this->writeJson($manifestPath, $manifest);

        $pagesDone[] = $page;
        sort($pagesDone);
        $status['pages_done'] = array_values(array_unique($pagesDone));
        $status['done_pages'] = count($status['pages_done']);
        $status['done_blocks'] = count($manifest);
        $pageCounts[$page] = $pageCount;
        $status['page_counts'] = $pageCounts;
        $status['total_blocks'] = array_sum($pageCounts);
        $status['updated_at'] = now()->toDateTimeString();
        if ($status['done_pages'] >= ($status['total_pages'] ?? 0)) {
            $status['completed_at'] = now()->toDateTimeString();
        }
        $this->writeJson($statusPath, $status);

        return $status;
    }

    public function getCropStatus(SourceFile $sourceFile, string $token): array
    {
        $baseDir = $this->cropBaseDir($sourceFile, $token);
        $status = $this->readJson($baseDir.'/status.json', []);
        if (! is_array($status)) {
            return [];
        }
        return $status;
    }

    public function getCropManifest(SourceFile $sourceFile, string $token): array
    {
        $baseDir = $this->cropBaseDir($sourceFile, $token);
        $manifest = $this->readJson($baseDir.'/manifest.json', []);
        return is_array($manifest) ? $manifest : [];
    }

    public function getLatestCropToken(SourceFile $sourceFile): ?string
    {
        $baseDir = storage_path('app/pdf_slip_crops/sf-'.$sourceFile->id);
        if (! is_dir($baseDir)) {
            return null;
        }

        $dirs = glob($baseDir.'/*', GLOB_ONLYDIR);
        if (! $dirs) {
            return null;
        }

        usort($dirs, function ($a, $b) {
            return filemtime($b) <=> filemtime($a);
        });

        return basename($dirs[0]);
    }

    private function normalizeOptions(array $options): array
    {
        return [
            'cache_version' => (int) config('votermaster.VM_PDF_SLIP_CACHE_VERSION', 1),
            'dpi' => (int) ($options['dpi'] ?? config('votermaster.VM_PDF_SLIP_DPI', 200)),
            'cols' => max(1, (int) ($options['cols'] ?? config('votermaster.VM_PDF_SLIP_COLS', 3))),
            'rows' => max(0, (int) ($options['rows'] ?? config('votermaster.VM_PDF_SLIP_ROWS', 5))),
            'auto_rows_min' => (int) ($options['auto_rows_min'] ?? config('votermaster.VM_PDF_SLIP_AUTO_ROWS_MIN', 5)),
            'auto_rows_max' => (int) ($options['auto_rows_max'] ?? config('votermaster.VM_PDF_SLIP_AUTO_ROWS_MAX', 6)),
            'margin_top' => $options['margin_top'] ?? config('votermaster.VM_PDF_SLIP_MARGIN_TOP', '0.16'),
            'margin_bottom' => $options['margin_bottom'] ?? config('votermaster.VM_PDF_SLIP_MARGIN_BOTTOM', '0.06'),
            'margin_left' => $options['margin_left'] ?? config('votermaster.VM_PDF_SLIP_MARGIN_LEFT', '0.03'),
            'margin_right' => $options['margin_right'] ?? config('votermaster.VM_PDF_SLIP_MARGIN_RIGHT', '0.03'),
            'inset' => (float) ($options['inset'] ?? config('votermaster.VM_PDF_SLIP_INSET', 0.002)),
            'box_height_scale' => (float) ($options['box_height_scale'] ?? config('votermaster.VM_PDF_SLIP_BOX_HEIGHT_SCALE', 1.0)),
            'box_pad' => (int) ($options['box_pad'] ?? config('votermaster.VM_PDF_SLIP_BOX_PAD', 6)),
            'detect_lines' => (bool) ($options['detect_lines'] ?? config('votermaster.VM_PDF_SLIP_DETECT_LINES', true)),
            'line_ratio' => (float) ($options['line_ratio'] ?? config('votermaster.VM_PDF_SLIP_LINE_RATIO', 0.65)),
            'line_threshold' => (int) ($options['line_threshold'] ?? config('votermaster.VM_PDF_SLIP_LINE_THRESHOLD', 130)),
            'line_gap' => (int) ($options['line_gap'] ?? config('votermaster.VM_PDF_SLIP_LINE_GAP', 12)),
            'scan_top' => (float) ($options['scan_top'] ?? config('votermaster.VM_PDF_SLIP_SCAN_TOP', 0.16)),
            'scan_bottom' => (float) ($options['scan_bottom'] ?? config('votermaster.VM_PDF_SLIP_SCAN_BOTTOM', 0.06)),
        ];
    }

    private function renderPageBlocks(string $filePath, int $page, array $options, string $baseDir): array
    {
        $manifestPath = $baseDir."/page-{$page}-manifest.json";
        if (file_exists($manifestPath)) {
            $manifest = json_decode((string) file_get_contents($manifestPath), true);
            if (is_array($manifest) && ! empty($manifest)) {
                $allExists = true;
                foreach ($manifest as $filename) {
                    if (! file_exists($baseDir.'/'.$filename)) {
                        $allExists = false;
                        break;
                    }
                }
                if ($allExists) {
                    return $manifest;
                }
            }
        }

        $tempDir = $baseDir.'/tmp-'.Str::random(6);
        File::ensureDirectoryExists($tempDir);

        $pageImage = $this->renderPageImage($filePath, $page, $options['dpi'], $tempDir);
        if (! $pageImage || ! file_exists($pageImage)) {
            throw new \RuntimeException('Page image generation failed.');
        }

        if (! function_exists('imagecreatefrompng')) {
            throw new \RuntimeException('GD extension required for slip crop.');
        }

        [$width, $height] = getimagesize($pageImage);
        $source = imagecreatefrompng($pageImage);
        $boxes = [];
        if (! empty($options['detect_lines'])) {
            $boxes = $this->detectNativeBoxes($source, $width, $height, $options);
            if (empty($boxes)) {
                $boxes = $this->detectGridBoxes($source, $width, $height, $options);
            }
        }
        if (empty($boxes)) {
            $gridOptions = $options;
            $gridOptions['rows'] = $this->resolveRows($options);
            $boxes = $this->buildGridBoxes($width, $height, $gridOptions);
        }
        $boxes = $this->filterBoxesByFrame($source, $boxes, $options);
        $files = [];
        foreach ($boxes as $index => $box) {
            $crop = imagecrop($source, $box);
            if (! $crop) {
                continue;
            }
            $target = $baseDir."/page-{$page}-block-{$index}.png";
            imagepng($crop, $target);
            imagedestroy($crop);
            $files[] = "page-{$page}-block-{$index}.png";
        }
        imagedestroy($source);

        File::deleteDirectory($tempDir);
        if (! empty($files)) {
            file_put_contents($manifestPath, json_encode($files));
        }

        return $files;
    }

    private function cropBaseDir(SourceFile $sourceFile, string $token): string
    {
        return storage_path('app/pdf_slip_crops/sf-'.$sourceFile->id.'/'.$token);
    }

    private function readJson(string $path, $default)
    {
        if (! file_exists($path)) {
            return $default;
        }

        $decoded = json_decode((string) file_get_contents($path), true);
        return is_array($decoded) ? $decoded : $default;
    }

    private function writeJson(string $path, array $payload): void
    {
        file_put_contents($path, json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
    }

    private function sortBlockFiles(array $files): array
    {
        usort($files, function ($a, $b) {
            return $this->extractBlockIndex($a) <=> $this->extractBlockIndex($b);
        });
        return $files;
    }

    private function extractBlockIndex(string $file): int
    {
        if (preg_match('/block-(\d+)\.png$/', $file, $matches)) {
            return (int) $matches[1];
        }
        return 0;
    }

    private function computeStartSerial(int $page, array $pageCounts, int $pageFrom, int $doneBlocks): int
    {
        if (! empty($pageCounts)) {
            $sum = 0;
            for ($p = $pageFrom; $p < $page; $p++) {
                $sum += (int) ($pageCounts[$p] ?? 0);
            }
            if ($sum > 0) {
                return $sum + 1;
            }
        }

        return max(1, $doneBlocks + 1);
    }

    private function inferExpectedBlockCount(array $pageCounts, array $options): ?int
    {
        $cols = (int) ($options['cols'] ?? 0);
        $minRows = (int) ($options['auto_rows_min'] ?? 0);
        $maxRows = (int) ($options['auto_rows_max'] ?? 0);
        if ($cols <= 0 || $minRows <= 0 || $maxRows <= 0) {
            return null;
        }

        $minCount = $cols * $minRows;
        $maxCount = $cols * $maxRows;
        $candidates = array_filter($pageCounts, function ($count) use ($minCount, $maxCount) {
            return in_array((int) $count, [$minCount, $maxCount], true);
        });
        if (count($candidates) < 2) {
            return null;
        }

        $freq = array_count_values(array_map('intval', $candidates));
        arsort($freq);
        $mode = (int) array_key_first($freq);
        $modeCount = (int) ($freq[$mode] ?? 0);
        if ($modeCount < 2) {
            return null;
        }

        return $mode;
    }

    private function purgePageCache(string $baseDir, int $page): void
    {
        $manifestPath = $baseDir."/page-{$page}-manifest.json";
        if (file_exists($manifestPath)) {
            @unlink($manifestPath);
        }

        $pattern = $baseDir."/page-{$page}-block-*.png";
        foreach (glob($pattern) ?: [] as $file) {
            @unlink($file);
        }
    }

    private function autoSelectRowLines(array $hLines, int $minRows, int $maxRows, int $scanTop, int $scanBottomY, int $height): array
    {
        $bestLines = $hLines;
        $bestRows = $minRows;
        $bestScore = null;

        for ($rows = $minRows; $rows <= $maxRows; $rows++) {
            $targetCount = $rows + 1;
            $candidate = $this->adjustLinesForMissingEdges($hLines, $targetCount, $scanTop, $scanBottomY, $height);
            $candidate = $this->selectBestLineSubset($candidate, $targetCount);
            $score = $this->scoreLineSubset($candidate, $scanTop, $scanBottomY);
            if ($score === null) {
                continue;
            }
            $score = $score - $this->rowBonusScore($candidate, $rows, $maxRows, $scanTop);
            if ($bestScore === null || $score < $bestScore - 0.0001 || (abs($score - $bestScore) <= 0.0001 && $rows > $bestRows)) {
                $bestScore = $score;
                $bestLines = $candidate;
                $bestRows = $rows;
            }
        }

        return [$bestLines, $bestRows];
    }

    private function rowBonusScore(array $lines, int $rows, int $maxRows, int $scanTop): float
    {
        if ($rows !== $maxRows || count($lines) < 2) {
            return 0.0;
        }

        $diffs = [];
        for ($i = 0; $i < count($lines) - 1; $i++) {
            $diffs[] = $lines[$i + 1] - $lines[$i];
        }
        $avg = array_sum($diffs) / max(1, count($diffs));
        if ($avg <= 0) {
            return 0.0;
        }
        $min = min($diffs);
        $firstGap = $lines[0] - $scanTop;

        // Favor max rows when spacing is uniform and top line aligns near scanTop.
        if (($min / $avg) > 0.6 && $firstGap >= 0 && $firstGap < ($avg * 0.6)) {
            return 0.003;
        }

        return 0.0;
    }

    private function scoreLineSubset(array $lines, int $scanTop, int $scanBottomY): ?float
    {
        if (count($lines) < 2) {
            return null;
        }
        $diffs = [];
        for ($i = 0; $i < count($lines) - 1; $i++) {
            $diffs[] = $lines[$i + 1] - $lines[$i];
        }
        $avg = array_sum($diffs) / max(1, count($diffs));
        if ($avg <= 0) {
            return null;
        }
        $variance = 0.0;
        foreach ($diffs as $d) {
            $variance += ($d - $avg) * ($d - $avg);
        }
        $variance /= max(1, count($diffs));
        $cv = sqrt($variance) / $avg;

        $span = end($lines) - $lines[0];
        $range = max(1, $scanBottomY - $scanTop);
        $coverage = $span / $range;
        $penalty = $coverage < 0.6 ? (0.6 - $coverage) * 0.5 : 0.0;

        return $cv + $penalty;
    }

    private function buildGridBoxes(int $width, int $height, array $options): array
    {
        $cols = $options['cols'];
        $rows = $this->resolveRows($options);
        $marginTop = $this->parseMarginValue($options['margin_top'], $height);
        $marginBottom = $this->parseMarginValue($options['margin_bottom'], $height);
        $marginLeft = $this->parseMarginValue($options['margin_left'], $width);
        $marginRight = $this->parseMarginValue($options['margin_right'], $width);

        $gridWidth = max(1, $width - $marginLeft - $marginRight);
        $gridHeight = max(1, $height - $marginTop - $marginBottom);

        $cellWidth = (int) floor($gridWidth / $cols);
        $cellHeight = (int) floor($gridHeight / $rows);

        $inset = max(0.0, min(0.1, (float) $options['inset']));
        $padX = (int) round($cellWidth * $inset);
        $padY = (int) round($cellHeight * $inset);

        $boxes = [];
        for ($row = 0; $row < $rows; $row++) {
            for ($col = 0; $col < $cols; $col++) {
                $x = $marginLeft + ($col * $cellWidth) + $padX;
                $y = $marginTop + ($row * $cellHeight) + $padY;
                $w = max(1, $cellWidth - ($padX * 2));
                $h = max(1, $cellHeight - ($padY * 2));
                $boxes[] = [
                    'x' => $x,
                    'y' => $y,
                    'width' => $w,
                    'height' => $h,
                ];
            }
        }

        return $boxes;
    }

    private function filterBoxesByFrame($image, array $boxes, array $options): array
    {
        if (empty($boxes)) {
            return $boxes;
        }

        $threshold = max(60, min(200, (int) ($options['line_threshold'] ?? 130)));
        $filtered = [];
        foreach ($boxes as $box) {
            if ($this->hasBoxFrame($image, $box, $threshold)) {
                $filtered[] = $box;
            }
        }

        $cols = max(1, (int) ($options['cols'] ?? 1));
        $rows = (int) ($options['rows'] ?? 0);
        if ($rows <= 0) {
            $rows = max(1, (int) ($options['auto_rows_min'] ?? 5));
        }
        $minExpected = $cols * $rows;

        // Only apply frame filter when enough likely voter boxes remain.
        if (count($filtered) >= $minExpected) {
            return $filtered;
        }

        return $boxes;
    }

    private function hasBoxFrame($image, array $box, int $darkLimit): bool
    {
        $x = (int) ($box['x'] ?? 0);
        $y = (int) ($box['y'] ?? 0);
        $w = (int) ($box['width'] ?? 0);
        $h = (int) ($box['height'] ?? 0);
        if ($w < 30 || $h < 30) {
            return false;
        }

        $leftX = min($x + 2, $x + $w - 1);
        $rightX = max($x, $x + $w - 3);
        $topY = min($y + 2, $y + $h - 1);

        $topRatio = $this->darkRatioHorizontal($image, $x, $x + $w - 1, $topY, $darkLimit);
        $leftRatio = $this->darkRatioVertical($image, $leftX, $y, $y + $h - 1, $darkLimit);
        $rightRatio = $this->darkRatioVertical($image, $rightX, $y, $y + $h - 1, $darkLimit);

        return $topRatio >= 0.45 && $leftRatio >= 0.35 && $rightRatio >= 0.35;
    }

    private function darkRatioHorizontal($image, int $x1, int $x2, int $y, int $darkLimit): float
    {
        $x1 = max(0, min($x1, imagesx($image) - 1));
        $x2 = max(0, min($x2, imagesx($image) - 1));
        $y = max(0, min($y, imagesy($image) - 1));
        if ($x2 < $x1) {
            return 0.0;
        }

        $dark = 0;
        $samples = 0;
        for ($x = $x1; $x <= $x2; $x += 2) {
            $rgb = imagecolorat($image, $x, $y);
            $r = ($rgb >> 16) & 0xFF;
            $g = ($rgb >> 8) & 0xFF;
            $b = $rgb & 0xFF;
            if (($r + $g + $b) < $darkLimit) {
                $dark++;
            }
            $samples++;
        }

        return $samples > 0 ? ($dark / $samples) : 0.0;
    }

    private function darkRatioVertical($image, int $x, int $y1, int $y2, int $darkLimit): float
    {
        $x = max(0, min($x, imagesx($image) - 1));
        $y1 = max(0, min($y1, imagesy($image) - 1));
        $y2 = max(0, min($y2, imagesy($image) - 1));
        if ($y2 < $y1) {
            return 0.0;
        }

        $dark = 0;
        $samples = 0;
        for ($y = $y1; $y <= $y2; $y += 2) {
            $rgb = imagecolorat($image, $x, $y);
            $r = ($rgb >> 16) & 0xFF;
            $g = ($rgb >> 8) & 0xFF;
            $b = $rgb & 0xFF;
            if (($r + $g + $b) < $darkLimit) {
                $dark++;
            }
            $samples++;
        }

        return $samples > 0 ? ($dark / $samples) : 0.0;
    }

    private function detectGridBoxes($image, int $width, int $height, array $options): array
    {
        $rows = (int) $options['rows'];
        if ($rows <= 0) {
            $rows = max(1, (int) ($options['auto_rows_max'] ?? 5));
        }
        $cols = (int) $options['cols'];
        $scanTop = $this->parseMarginValue($options['scan_top'], $height);
        $scanBottom = $this->parseMarginValue($options['scan_bottom'], $height);
        $scanTop = max(0, min($height - 1, $scanTop));
        $scanBottom = max(0, min($height - 1, $scanBottom));
        $scanBottomY = max($scanTop + 1, $height - $scanBottom);

        $vLines = $this->detectLines($image, $width, $height, false, $options, $scanTop, $scanBottomY);
        $hLines = $this->detectLines($image, $width, $height, true, $options, $scanTop, $scanBottomY);
        if (count($vLines) >= 2) {
            $xStart = (int) $vLines[0];
            $xEnd = (int) $vLines[count($vLines) - 1];
            if ($xEnd > $xStart) {
                $hLinesGrid = $this->detectHorizontalLinesInRange($image, $xStart, $xEnd, $height, $options, $scanTop, $scanBottomY);
                if (count($hLinesGrid) >= 2) {
                    $hLines = $hLinesGrid;
                }
            }
        }

        if (count($hLines) < $rows + 1 || count($vLines) < $cols + 1) {
            return [];
        }

        $hLines = $this->selectGridLines($hLines, $rows + 1);
        $vLines = $this->selectGridLines($vLines, $cols + 1);
        if (empty($hLines) || empty($vLines)) {
            return [];
        }

        return $this->buildBoxesFromLines(
            $hLines,
            $vLines,
            $options['inset'],
            (float) ($options['box_height_scale'] ?? 1.0),
            (int) ($options['box_pad'] ?? 0),
            $width,
            $height
        );
    }

    private function detectNativeBoxes($image, int $width, int $height, array $options): array
    {
        $scanTop = $this->parseMarginValue($options['scan_top'], $height);
        $scanBottom = $this->parseMarginValue($options['scan_bottom'], $height);
        $scanTop = max(0, min($height - 1, $scanTop));
        $scanBottom = max(0, min($height - 1, $scanBottom));
        $scanBottomY = max($scanTop + 1, $height - $scanBottom);

        $hLines = $this->detectLines($image, $width, $height, true, $options, $scanTop, $scanBottomY);
        $vLines = $this->detectLines($image, $width, $height, false, $options, $scanTop, $scanBottomY);

        if (count($hLines) < 2 || count($vLines) < 2) {
            return [];
        }

        sort($hLines);
        sort($vLines);

        $targetRows = (int) ($options['rows'] ?? 0);
        $targetCols = (int) ($options['cols'] ?? 0);
        if ($targetCols > 0) {
            $vLines = $this->selectGridLines($vLines, $targetCols + 1);
        }

        if ($targetRows > 0) {
            $targetCount = $targetRows + 1;
            $hLines = $this->adjustLinesForMissingEdges($hLines, $targetCount, $scanTop, $scanBottomY, $height);
            $hLines = $this->selectGridLines($hLines, $targetCount);
        } else {
            $minRows = max(1, (int) ($options['auto_rows_min'] ?? 5));
            $maxRows = max($minRows, (int) ($options['auto_rows_max'] ?? $minRows));
            $pairBoxes = $this->buildBoxesFromLinePairs(
                $image,
                $hLines,
                $vLines,
                $targetCols,
                (float) $options['inset'],
                (float) ($options['box_height_scale'] ?? 1.0),
                (int) ($options['box_pad'] ?? 0),
                $width,
                $height,
                $minRows,
                $maxRows,
                (int) ($options['line_threshold'] ?? 130)
            );
            if (! empty($pairBoxes)) {
                return $pairBoxes;
            }
            [$hLines, $targetRows] = $this->autoSelectRowLines($hLines, $minRows, $maxRows, $scanTop, $scanBottomY, $height);
        }

        if (count($hLines) < 2 || count($vLines) < 2) {
            return [];
        }

        $inset = (float) $options['inset'];
        $heightScale = (float) ($options['box_height_scale'] ?? 1.0);
        $boxPad = (int) ($options['box_pad'] ?? 0);
        $boxes = $this->buildBoxesFromLines($hLines, $vLines, $inset, $heightScale, $boxPad, $width, $height);
        if ($targetRows > 0 && $targetCols > 0 && count($boxes) !== $targetRows * $targetCols) {
            return [];
        }
        return $boxes;
    }

    private function buildBoxesFromLinePairs(
        $image,
        array $horizontal,
        array $vertical,
        int $targetCols,
        float $inset,
        float $heightScale,
        int $boxPad,
        int $imageWidth,
        int $imageHeight,
        int $minRows,
        int $maxRows,
        int $lineThreshold
    ): array {
        if (count($horizontal) < 4 || count($vertical) < 2 || $targetCols <= 0) {
            return [];
        }

        sort($horizontal);
        sort($vertical);
        $vertical = $this->selectGridLines($vertical, $targetCols + 1);
        if (count($vertical) < ($targetCols + 1)) {
            return [];
        }

        $diffs = [];
        for ($i = 0; $i < count($horizontal) - 1; $i++) {
            $diffs[] = $horizontal[$i + 1] - $horizontal[$i];
        }
        if (empty($diffs)) {
            return [];
        }

        $median = $this->median($diffs);
        if ($median <= 0) {
            return [];
        }

        $spanThreshold = max(140.0, $median * 0.95);
        $pairs = [];
        for ($i = 0; $i < count($horizontal) - 1; $i++) {
            $top = (int) $horizontal[$i];
            $bottom = (int) $horizontal[$i + 1];
            $span = $bottom - $top;
            if ($span < $spanThreshold) {
                continue;
            }

            // Some pages have an extra short gap before the true row bottom line.
            if (isset($horizontal[$i + 2])) {
                $nextGap = (int) $horizontal[$i + 2] - $bottom;
                if ($nextGap > 0 && $nextGap <= max(80, (int) round($span * 0.55))) {
                    $bottom = (int) $horizontal[$i + 2];
                    $span = $bottom - $top;
                }
            }

            if (! $this->rowHasStrongVerticalFrames($image, $vertical, $top, $bottom, $lineThreshold)) {
                continue;
            }
            $pairs[] = [$top, $bottom];
        }

        if (count($pairs) < $minRows && count($horizontal) >= ($minRows * 2)) {
            $pairs = [];
            for ($i = 0; ($i + 1) < count($horizontal); $i += 2) {
                $pairs[] = [$horizontal[$i], $horizontal[$i + 1]];
            }
        }

        if (count($pairs) < $minRows || count($pairs) > $maxRows) {
            return [];
        }

        $inset = max(0.0, min(0.2, $inset));
        $boxes = [];
        foreach ($pairs as $pair) {
            [$top, $bottom] = $pair;
            $rowHeight = max(1, (int) ($bottom - $top));
            for ($col = 0; $col < $targetCols; $col++) {
                $x = (int) $vertical[$col];
                $y = (int) $top;
                $w = (int) ($vertical[$col + 1] - $x);
                $h = $rowHeight;
                if ($heightScale > 1.0) {
                    $extra = (int) round($h * ($heightScale - 1.0));
                    $h = min($rowHeight, $h + $extra);
                }

                $pad = max(0, min(20, $boxPad));
                if ($pad > 0) {
                    $padX = $pad;
                    $padY = 0;
                    $x = max(0, $x - $padX);
                    $y = max(0, $y - $padY);
                    $w = $w + ($padX * 2);
                    $h = $h + ($padY * 2);
                    if ($imageWidth > 0) {
                        $w = min($imageWidth - $x, $w);
                    }
                    if ($imageHeight > 0) {
                        $h = min($imageHeight - $y, $h);
                    }
                }

                if ($w <= 1 || $h <= 1) {
                    continue;
                }

                $insetX = (int) floor($w * $inset);
                $insetY = (int) floor($h * $inset);
                $x += $insetX;
                $y += $insetY;
                $w = max(1, $w - (2 * $insetX));
                $h = max(1, $h - (2 * $insetY));

                $boxes[] = [
                    'x' => $x,
                    'y' => $y,
                    'width' => $w,
                    'height' => $h,
                ];
            }
        }

        $boxes = $this->trimOverlappingBoxes($boxes, $targetCols, $imageHeight, $image, $vertical, $lineThreshold);

        return $boxes;
    }

    private function trimOverlappingBoxes(
        array $boxes,
        int $cols,
        int $imageHeight,
        $image,
        array $verticalLines,
        int $darkLimit
    ): array
    {
        if ($cols <= 0 || empty($boxes)) {
            return $boxes;
        }

        $rows = (int) floor(count($boxes) / $cols);
        if ($rows <= 1) {
            return $boxes;
        }

        for ($row = 0; $row < $rows - 1; $row++) {
            $startA = $row * $cols;
            $startB = ($row + 1) * $cols;
            if (! isset($boxes[$startA], $boxes[$startB])) {
                continue;
            }

            $rowBottom = PHP_INT_MIN;
            for ($c = 0; $c < $cols; $c++) {
                $idx = $startA + $c;
                if (! isset($boxes[$idx])) {
                    continue;
                }
                $rowBottom = max($rowBottom, $boxes[$idx]['y'] + $boxes[$idx]['height']);
            }

            $nextTop = PHP_INT_MAX;
            for ($c = 0; $c < $cols; $c++) {
                $idx = $startB + $c;
                if (! isset($boxes[$idx])) {
                    continue;
                }
                $nextTop = min($nextTop, $boxes[$idx]['y']);
            }

            if ($rowBottom > $nextTop) {
                $boundaryY = $this->findBoundaryY($image, $verticalLines, $nextTop, $darkLimit, $imageHeight);
                $allowedBottom = max(0, $boundaryY - 2);
                for ($c = 0; $c < $cols; $c++) {
                    $idx = $startA + $c;
                    if (! isset($boxes[$idx])) {
                        continue;
                    }
                    $newHeight = $allowedBottom - $boxes[$idx]['y'];
                    $boxes[$idx]['height'] = max(1, min($newHeight, $imageHeight - $boxes[$idx]['y']));
                }
            }
        }

        return $boxes;
    }

    private function findBoundaryY($image, array $verticalLines, int $guessY, int $darkLimit, int $imageHeight): int
    {
        $bestY = max(0, min($imageHeight - 1, $guessY));
        if (! $image || empty($verticalLines)) {
            return $bestY;
        }
        $bestScore = 0.0;
        $from = max(0, $bestY - 20);
        $to = min($imageHeight - 1, $bestY + 20);

        for ($y = $from; $y <= $to; $y++) {
            $minRatio = 1.0;
            foreach ($verticalLines as $x) {
                $ratio = $this->darkRatioHorizontal($image, max(0, (int) $x - 8), min(imagesx($image) - 1, (int) $x + 8), $y, $darkLimit);
                $minRatio = min($minRatio, $ratio);
            }
            if ($minRatio > $bestScore) {
                $bestScore = $minRatio;
                $bestY = $y;
            }
        }

        return $bestY;
    }

    private function rowHasStrongVerticalFrames($image, array $verticalLines, int $yTop, int $yBottom, int $darkLimit): bool
    {
        if (count($verticalLines) < 2 || $yBottom <= $yTop) {
            return false;
        }

        $scores = [];
        foreach ($verticalLines as $x) {
            $scores[] = $this->maxDarkRatioVerticalAround($image, (int) $x, $yTop, $yBottom, $darkLimit);
        }

        $avg = array_sum($scores) / max(1, count($scores));
        $min = min($scores);

        return $avg >= 0.75 && $min >= 0.45;
    }

    private function maxDarkRatioVerticalAround($image, int $x, int $y1, int $y2, int $darkLimit): float
    {
        $best = 0.0;
        for ($offset = -6; $offset <= 6; $offset++) {
            $best = max($best, $this->darkRatioVertical($image, $x + $offset, $y1, $y2, $darkLimit));
        }

        return $best;
    }

    private function detectLines($image, int $width, int $height, bool $horizontal, array $options, int $scanTop, int $scanBottomY): array
    {
        $step = 2;
        $ratio = max(0.02, min(0.3, (float) $options['line_ratio']));
        $darkLimit = max(60, min(200, (int) $options['line_threshold']));
        $lines = [];

        if ($horizontal) {
            for ($y = $scanTop; $y < $scanBottomY; $y++) {
                $dark = 0;
                $samples = 0;
                for ($x = 0; $x < $width; $x += $step) {
                    $rgb = imagecolorat($image, $x, $y);
                    $r = ($rgb >> 16) & 0xFF;
                    $g = ($rgb >> 8) & 0xFF;
                    $b = $rgb & 0xFF;
                    if (($r + $g + $b) < $darkLimit) {
                        $dark++;
                    }
                    $samples++;
                }
                if ($samples > 0 && ($dark / $samples) >= $ratio) {
                    $lines[] = $y;
                }
            }
        } else {
            for ($x = 0; $x < $width; $x++) {
                $dark = 0;
                $samples = 0;
                for ($y = $scanTop; $y < $scanBottomY; $y += $step) {
                    $rgb = imagecolorat($image, $x, $y);
                    $r = ($rgb >> 16) & 0xFF;
                    $g = ($rgb >> 8) & 0xFF;
                    $b = $rgb & 0xFF;
                    if (($r + $g + $b) < $darkLimit) {
                        $dark++;
                    }
                    $samples++;
                }
                if ($samples > 0 && ($dark / $samples) >= $ratio) {
                    $lines[] = $x;
                }
            }
        }

        return $this->clusterLines($lines, (int) $options['line_gap']);
    }

    private function detectHorizontalLinesInRange($image, int $xStart, int $xEnd, int $height, array $options, int $scanTop, int $scanBottomY): array
    {
        $step = 2;
        $ratio = max(0.02, min(0.3, (float) $options['line_ratio']));
        $darkLimit = max(60, min(200, (int) $options['line_threshold']));
        $lines = [];
        $xStart = max(0, $xStart);
        $xEnd = max($xStart + 1, $xEnd);

        for ($y = $scanTop; $y < $scanBottomY; $y++) {
            $dark = 0;
            $samples = 0;
            for ($x = $xStart; $x < $xEnd; $x += $step) {
                $rgb = imagecolorat($image, $x, $y);
                $r = ($rgb >> 16) & 0xFF;
                $g = ($rgb >> 8) & 0xFF;
                $b = $rgb & 0xFF;
                if (($r + $g + $b) < $darkLimit) {
                    $dark++;
                }
                $samples++;
            }
            if ($samples > 0 && ($dark / $samples) >= $ratio) {
                $lines[] = $y;
            }
        }

        return $this->clusterLines($lines, (int) $options['line_gap']);
    }

    private function clusterLines(array $lines, int $gap): array
    {
        if (empty($lines)) {
            return [];
        }

        sort($lines);
        $gap = max(1, $gap);
        $clusters = [];
        $current = [$lines[0]];

        for ($i = 1; $i < count($lines); $i++) {
            $line = $lines[$i];
            $last = $current[count($current) - 1];
            if (($line - $last) <= $gap) {
                $current[] = $line;
            } else {
                $clusters[] = (int) round(array_sum($current) / count($current));
                $current = [$line];
            }
        }

        if (! empty($current)) {
            $clusters[] = (int) round(array_sum($current) / count($current));
        }

        return $clusters;
    }

    private function selectGridLines(array $lines, int $count): array
    {
        if (count($lines) < $count) {
            return [];
        }

        sort($lines);
        $best = [];
        $bestSpan = -1;
        $bestStart = PHP_INT_MAX;
        $tieEpsilon = 8;

        for ($i = 0; $i <= count($lines) - $count; $i++) {
            $subset = array_slice($lines, $i, $count);
            $span = end($subset) - $subset[0];
            if ($span > ($bestSpan + $tieEpsilon)) {
                $bestSpan = $span;
                $best = $subset;
                $bestStart = $subset[0];
            } elseif ($bestSpan >= 0 && abs($span - $bestSpan) <= $tieEpsilon && $subset[0] < $bestStart) {
                $best = $subset;
                $bestStart = $subset[0];
            }
        }

        return $best;
    }

    private function buildBoxesFromLines(
        array $horizontal,
        array $vertical,
        float $inset,
        float $heightScale = 1.0,
        int $boxPad = 0,
        int $imageWidth = 0,
        int $imageHeight = 0
    ): array
    {
        $rows = count($horizontal) - 1;
        $cols = count($vertical) - 1;
        if ($rows <= 0 || $cols <= 0) {
            return [];
        }

        $inset = max(0.0, min(0.2, $inset));
        $boxes = [];
        for ($row = 0; $row < $rows; $row++) {
            for ($col = 0; $col < $cols; $col++) {
                $x = (int) $vertical[$col];
                $y = (int) $horizontal[$row];
                $w = (int) ($vertical[$col + 1] - $x);
                $h = (int) ($horizontal[$row + 1] - $y);
                $maxHeight = (int) ($horizontal[$row + 1] - $horizontal[$row]);
                if ($heightScale > 1.0) {
                    $extra = (int) round($h * ($heightScale - 1.0));
                    $h = min($maxHeight, $h + $extra);
                }

                $pad = max(0, min(20, $boxPad));
                if ($pad > 0) {
                    $padX = $pad;
                    $padY = 0;
                    // Avoid vertical bleed into previous/next voter block.
                    $x = max(0, $x - $padX);
                    $y = max(0, $y - $padY);
                    $w = $w + ($padX * 2);
                    $h = $h + ($padY * 2);
                    if ($imageWidth > 0) {
                        $w = min($imageWidth - $x, $w);
                    }
                    if ($imageHeight > 0) {
                        $h = min($imageHeight - $y, $h);
                    }
                }

                if ($w <= 1 || $h <= 1) {
                    continue;
                }

                $insetX = (int) floor($w * $inset);
                $insetY = (int) floor($h * $inset);
                $x = $x + $insetX;
                $y = $y + $insetY;
                $w = max(1, $w - (2 * $insetX));
                $h = max(1, $h - (2 * $insetY));

                $boxes[] = [
                    'x' => $x,
                    'y' => $y,
                    'width' => $w,
                    'height' => $h,
                ];
            }
        }

        return $boxes;
    }

    private function buildBoxesFromLinesFiltered(array $horizontal, array $vertical, array $rowIndices, float $inset): array
    {
        $cols = count($vertical) - 1;
        if (empty($rowIndices) || $cols <= 0) {
            return [];
        }

        $inset = max(0.0, min(0.2, $inset));
        $boxes = [];
        foreach ($rowIndices as $row) {
            if (! isset($horizontal[$row], $horizontal[$row + 1])) {
                continue;
            }
            for ($col = 0; $col < $cols; $col++) {
                $x = (int) $vertical[$col];
                $y = (int) $horizontal[$row];
                $w = (int) ($vertical[$col + 1] - $x);
                $h = (int) ($horizontal[$row + 1] - $y);

                if ($w <= 1 || $h <= 1) {
                    continue;
                }

                $insetX = (int) floor($w * $inset);
                $insetY = (int) floor($h * $inset);
                $x = $x + $insetX;
                $y = $y + $insetY;
                $w = max(1, $w - (2 * $insetX));
                $h = max(1, $h - (2 * $insetY));

                $boxes[] = [
                    'x' => $x,
                    'y' => $y,
                    'width' => $w,
                    'height' => $h,
                ];
            }
        }

        return $boxes;
    }

    private function median(array $values): float
    {
        if (empty($values)) {
            return 0.0;
        }

        sort($values);
        $count = count($values);
        $mid = (int) floor($count / 2);
        if ($count % 2 === 0) {
            return ($values[$mid - 1] + $values[$mid]) / 2;
        }

        return (float) $values[$mid];
    }

    private function selectBestLineSubset(array $lines, int $targetCount): array
    {
        if ($targetCount <= 1 || count($lines) <= $targetCount) {
            return $lines;
        }

        $best = [];
        $bestScore = null;
        $bestSpan = -1;
        $minLine = (int) min($lines);
        $n = count($lines);
        $indices = range(0, $n - 1);

        $combine = function ($start, $left, $picked) use (&$combine, $lines, $targetCount, &$best, &$bestScore, &$bestSpan, $n, $minLine) {
            if ($left === 0) {
                $subset = array_map(fn ($i) => $lines[$i], $picked);
                $diffs = [];
                for ($i = 0; $i < count($subset) - 1; $i++) {
                    $diffs[] = $subset[$i + 1] - $subset[$i];
                }
                $avg = array_sum($diffs) / max(1, count($diffs));
                if ($avg <= 0) {
                    return;
                }
                $variance = 0.0;
                foreach ($diffs as $d) {
                    $variance += ($d - $avg) * ($d - $avg);
                }
                $variance /= max(1, count($diffs));
                $cv = sqrt($variance) / $avg;
                $span = end($subset) - $subset[0];
                $spanSafe = max(1, $span);
                $startPenalty = max(0.0, (($subset[0] - $minLine) / $spanSafe));
                // Prefer earlier-starting subsets when spacing quality is similar.
                $score = $cv + ($startPenalty * 0.15);

                if ($bestScore === null || $score < $bestScore - 0.0001 || (abs($score - $bestScore) <= 0.0001 && $span > $bestSpan)) {
                    $bestScore = $score;
                    $bestSpan = $span;
                    $best = $subset;
                }
                return;
            }
            for ($i = $start; $i <= $n - $left; $i++) {
                $combine($i + 1, $left - 1, array_merge($picked, [$i]));
            }
        };

        $combine(0, $targetCount, []);

        return ! empty($best) ? $best : array_slice($lines, 0, $targetCount);
    }

    private function adjustLinesForMissingEdges(
        array $lines,
        int $targetCount,
        int $scanTop,
        int $scanBottomY,
        int $imageHeight
    ): array {
        if ($targetCount <= 1 || count($lines) < 2) {
            return $lines;
        }

        sort($lines);
        $diffs = [];
        for ($i = 0; $i < count($lines) - 1; $i++) {
            $diffs[] = $lines[$i + 1] - $lines[$i];
        }
        $avgGap = $this->median($diffs);
        if ($avgGap <= 0) {
            return $lines;
        }

        $candidates = $lines;
        $first = $lines[0];
        $last = $lines[count($lines) - 1];
        $edgeThreshold = $avgGap * 0.8;

        if (count($lines) === $targetCount && count($lines) >= 2) {
            $topGap = $lines[0] - $scanTop;
            if ($topGap > 0 && $topGap < ($avgGap * 0.7)) {
                $lines[0] = (int) round($scanTop);
                sort($lines);
                return $lines;
            }

            $bottomGap = $scanBottomY - $lines[count($lines) - 1];
            if ($bottomGap > 0 && $bottomGap < ($avgGap * 0.7)) {
                $lines[count($lines) - 1] = (int) round($scanBottomY);
                sort($lines);
                return $lines;
            }
        }

        if (($first - $scanTop) > $edgeThreshold) {
            $newLine = (int) round($scanTop);
            if ($newLine > 0) {
                $candidates[] = $newLine;
            }
        }
        if (($scanBottomY - $last) > $edgeThreshold) {
            $newLine = (int) round($scanBottomY);
            if ($newLine < $imageHeight) {
                $candidates[] = $newLine;
            }
        }

        sort($candidates);
        if (count($candidates) > $targetCount) {
            return $this->selectBestLineSubset($candidates, $targetCount);
        }

        return $candidates;
    }

    private function renderPageImage(string $filePath, int $page, int $dpi, string $tempDir): string
    {
        $pdftoppm = $this->resolveBinary(config('votermaster.VM_PDFTOPPM_PATH', 'pdftoppm'));
        $prefix = $tempDir.'/page';
        $process = new Process([
            $pdftoppm,
            '-r',
            (string) $dpi,
            '-f',
            (string) $page,
            '-l',
            (string) $page,
            '-png',
            $filePath,
            $prefix,
        ]);
        $process->setWorkingDirectory(base_path());
        $env = array_merge($_ENV, $_SERVER);
        $binDir = '';
        if ($pdftoppm !== '' && file_exists($pdftoppm)) {
            $binDir = dirname($pdftoppm);
        }
        if ($binDir !== '') {
            $env['PATH'] = $binDir.PATH_SEPARATOR.($env['PATH'] ?? '');
            $process->setWorkingDirectory($binDir);
        }
        $process->setEnv($env);
        $process->setTimeout(600);
        $process->run();
        if (! $process->isSuccessful()) {
            $error = trim($process->getErrorOutput());
            $code = $process->getExitCode();
            $message = $error !== '' ? $error : 'PDF to image conversion failed.';
            $message .= $code !== null ? " (exit code: {$code})" : '';
            throw new \RuntimeException($message);
        }

        $expected = $prefix.'-'.$page.'.png';
        if (file_exists($expected)) {
            return $expected;
        }

        $matches = glob($prefix.'-*.png');
        if ($matches) {
            return $matches[0];
        }

        throw new \RuntimeException('Page image not found.');
    }

    private function resolveBinary(string $binary): string
    {
        if ($binary === '') {
            return $binary;
        }

        $isAbsolute = str_contains($binary, ':') || str_starts_with($binary, DIRECTORY_SEPARATOR);

        if (! $isAbsolute) {
            $candidate = base_path($binary);
            if (file_exists($candidate)) {
                return $candidate;
            }
        }

        if (file_exists($binary)) {
            return $binary;
        }

        if ($isAbsolute) {
            return basename(str_replace('\\', '/', $binary));
        }

        return $binary;
    }

    private function resolveRows(array $options): int
    {
        $rows = (int) ($options['rows'] ?? 0);
        if ($rows > 0) {
            return $rows;
        }

        $autoMin = max(1, (int) ($options['auto_rows_min'] ?? 5));
        $autoMax = max($autoMin, (int) ($options['auto_rows_max'] ?? $autoMin));

        return max(1, min($autoMin, $autoMax));
    }

    private function resolveStoredPath(?string $storedPath): ?string
    {
        if (! $storedPath) {
            return null;
        }

        if (file_exists($storedPath)) {
            return $storedPath;
        }

        try {
            $resolved = Storage::path($storedPath);
            if (file_exists($resolved)) {
                return $resolved;
            }
        } catch (\Throwable $e) {
            // fall through
        }

        $fallback = storage_path('app/'.$storedPath);
        if (file_exists($fallback)) {
            return $fallback;
        }

        $legacy = storage_path('app/private/'.$storedPath);
        if (file_exists($legacy)) {
            return $legacy;
        }

        return null;
    }

    private function parseMarginValue($raw, int $total): int
    {
        if ($raw === null || $raw === '') {
            return 0;
        }

        $value = trim((string) $raw);
        if ($value === '') {
            return 0;
        }

        if (str_ends_with($value, '%')) {
            $percent = (float) rtrim($value, '%');
            $percent = max(0.0, min(30.0, $percent));

            return (int) round($total * ($percent / 100));
        }

        $number = (float) $value;
        if ($number > 1) {
            return (int) round($number);
        }

        $number = max(0.0, min(0.3, $number));

        return (int) round($total * $number);
    }
}
