<?php

namespace App\Queries;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;

class VoterFilters
{
    public static function apply(Builder $query, array $filters): Builder
    {
        if (! empty($filters['q'])) {
            $search = $filters['q'];
            $query->where(function (Builder $builder) use ($search) {
                $builder->where('name_bn', 'like', "%{$search}%")
                    ->orWhere('voter_no', 'like', "%{$search}%")
                    ->orWhere('father_name_bn', 'like', "%{$search}%")
                    ->orWhere('mother_name_bn', 'like', "%{$search}%")
                    ->orWhere('address_bn', 'like', "%{$search}%");
            });
        }

        if (! empty($filters['area_id'])) {
            $query->where('area_id', $filters['area_id']);
        }

        if (! empty($filters['profession_bn'])) {
            $query->where('profession_bn', $filters['profession_bn']);
        }

        if (! empty($filters['date_of_birth'])) {
            $normalizedDob = self::normalizeDob($filters['date_of_birth']);
            if ($normalizedDob) {
                $query->whereDate('date_of_birth', $normalizedDob);
            }
        }

        if (! empty($filters['gender'])) {
            $query->where('gender', $filters['gender']);
        }

        if (! empty($filters['source_file_id'])) {
            $query->where('source_file_id', $filters['source_file_id']);
        }

        if (! empty($filters['min_age']) || ! empty($filters['max_age'])) {
            $today = Carbon::today();

            if (! empty($filters['min_age'])) {
                $maxDob = $today->copy()->subYears((int) $filters['min_age'])->toDateString();
                $query->whereDate('date_of_birth', '<=', $maxDob);
            }

            if (! empty($filters['max_age'])) {
                $minDob = $today->copy()->subYears((int) $filters['max_age'] + 1)->addDay()->toDateString();
                $query->whereDate('date_of_birth', '>=', $minDob);
            }
        }

        return $query;
    }

    private static function normalizeDob(string $value): ?string
    {
        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $value = strtr($value, [
            '০' => '0', '১' => '1', '২' => '2', '৩' => '3', '৪' => '4',
            '৫' => '5', '৬' => '6', '৭' => '7', '৮' => '8', '৯' => '9',
        ]);
        $value = str_replace(['.', ' '], '/', $value);

        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return $value;
        }

        if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $value)) {
            [$day, $month, $year] = explode('/', $value);
            return sprintf('%04d-%02d-%02d', (int) $year, (int) $month, (int) $day);
        }

        if (preg_match('/^\d{2}-\d{2}-\d{4}$/', $value)) {
            [$day, $month, $year] = explode('-', $value);
            return sprintf('%04d-%02d-%02d', (int) $year, (int) $month, (int) $day);
        }

        return null;
    }
}
