<?php

namespace App\Jobs;

use App\Models\SourceFile;
use App\Services\VoterPdfImportService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ImportSourceFilePage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $timeout = 1200;

    public function __construct(
        public int $sourceFileId,
        public int $page,
        public ?string $genderOverride = null,
        public ?string $importToken = null
    ) {
        $this->onQueue('imports');
        $this->onConnection(env('QUEUE_CONNECTION', 'database'));
    }

    public function handle(VoterPdfImportService $service): void
    {
        $sourceFile = SourceFile::find($this->sourceFileId);
        if (! $sourceFile) {
            return;
        }

        if (in_array($sourceFile->import_status, ['failed', 'stopped', 'completed'], true)) {
            return;
        }

        $currentToken = $sourceFile->import_options['import_token'] ?? null;
        if ($currentToken && $this->importToken !== $currentToken) {
            return;
        }

        try {
            $service->importPage($sourceFile, $this->page, $this->genderOverride);

            DB::transaction(function () use ($sourceFile) {
                $sourceFile->refresh();

                if (in_array($sourceFile->import_status, ['failed', 'stopped'], true)) {
                    return;
                }

                if ($sourceFile->import_status === 'queued') {
                    $sourceFile->update(['import_status' => 'running']);
                }

                $sourceFile->increment('imported_pages');
                $sourceFile->refresh();

                if ($sourceFile->total_pages && $sourceFile->imported_pages >= $sourceFile->total_pages) {
                    $sourceFile->update([
                        'import_status' => 'completed',
                        'import_finished_at' => now(),
                    ]);

                    if ($sourceFile->area) {
                        $sourceFile->area->update(['total_voters' => $sourceFile->area->voters()->count()]);
                    }
                }
            });
        } catch (\Throwable $e) {
            $sourceFile->update([
                'import_status' => 'failed',
                'import_finished_at' => now(),
            ]);

            Log::error('Source file page import failed', [
                'source_file_id' => $this->sourceFileId,
                'page' => $this->page,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }
}
