<?php

namespace App\Jobs;

use App\Models\PdfExport;
use App\Services\PdfSlipPrintService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;

class GenerateCandidateSlipPdf implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(private int $exportId)
    {
        $this->onQueue('pdf');
    }

    public function handle(PdfSlipPrintService $service): void
    {
        $export = PdfExport::with(['sourceFile', 'candidate'])->find($this->exportId);
        if (! $export) {
            return;
        }

        if ($export->status === 'canceled') {
            return;
        }

        $export->update([
            'status' => 'running',
            'started_at' => now(),
            'progress' => 1,
        ]);

        $token = $service->getLatestCropToken($export->sourceFile);
        if (! $token) {
            $export->update([
                'status' => 'failed',
                'message' => 'Crop data not found for this source file.',
            ]);
            return;
        }

        $manifest = $service->getCropManifest($export->sourceFile, $token);
        if (empty($manifest)) {
            $export->update([
                'status' => 'failed',
                'message' => 'Crop data not found for this source file.',
            ]);
            return;
        }

        $serialFrom = (int) ($export->serial_from ?? 1);
        $serialTo = $export->serial_to ? (int) $export->serial_to : PHP_INT_MAX;
        if ($serialTo < $serialFrom) {
            $serialTo = $serialFrom;
        }

        $filtered = array_values(array_filter($manifest, function ($entry) use ($serialFrom, $serialTo) {
            $serial = (int) ($entry['serial'] ?? 0);
            return $serial >= $serialFrom && $serial <= $serialTo;
        }));

        $total = count($filtered);
        if ($total === 0) {
            $export->update([
                'status' => 'failed',
                'message' => 'No slips found in selected range.',
            ]);
            return;
        }

        $export->update([
            'total_slips' => $total,
            'processed_slips' => 0,
            'progress' => 5,
        ]);

        $flatFiles = [];
        $processed = 0;
        foreach (array_chunk($filtered, 10) as $chunk) {
            $export->refresh();
            if ($export->status === 'canceled') {
                return;
            }

            foreach ($chunk as $entry) {
                $file = $entry['file'] ?? null;
                $path = $file ? storage_path('app/pdf_slip_crops/sf-'.$export->source_file_id.'/'.$token.'/'.$file) : null;
                $flatFiles[] = [
                    'serial' => $entry['serial'] ?? null,
                    'file' => $file,
                    'image_url' => $path ? 'file://'.str_replace('\\', '/', $path) : null,
                ];
                $processed++;
            }

            $progress = (int) min(80, round(($processed / $total) * 80));
            $export->update([
                'processed_slips' => $processed,
                'progress' => max(5, $progress),
            ]);
        }

        $export->refresh();
        if ($export->status === 'canceled') {
            return;
        }

        $candidate = $export->candidate;
        Storage::disk('local')->makeDirectory('fonts');
        $photoPath = $candidate?->photo_path && Storage::disk('public')->exists($candidate->photo_path)
            ? 'file://'.str_replace('\\', '/', Storage::disk('public')->path($candidate->photo_path))
            : null;
        $symbolPath = $candidate?->symbol_path && Storage::disk('public')->exists($candidate->symbol_path)
            ? 'file://'.str_replace('\\', '/', Storage::disk('public')->path($candidate->symbol_path))
            : null;

        $pdf = Pdf::loadView('admin.candidate_slip_prints.pdf', [
            'sourceFile' => $export->sourceFile,
            'candidate' => $candidate,
            'flatFiles' => $flatFiles,
            'perPage' => 10,
            'slipHeader' => \App\Models\Setting::getValue('slip_header_bn'),
            'slipFooter' => \App\Models\Setting::getValue('slip_footer_bn'),
            'candidatePhotoUrl' => $photoPath,
            'candidateSymbolUrl' => $symbolPath,
        ])->setPaper('a4')
            ->setOption('defaultFont', 'NotoSansBengali')
            ->setOption('isRemoteEnabled', true)
            ->setOption('fontDir', public_path('fonts'))
            ->setOption('fontCache', storage_path('fonts'));

        $export->refresh();
        if ($export->status === 'canceled') {
            return;
        }

        $fileName = 'candidate-slip-'.$export->id.'.pdf';
        $relative = 'pdf_exports/'.$fileName;
        Storage::disk('local')->put($relative, $pdf->output());

        $export->update([
            'status' => 'done',
            'progress' => 100,
            'file_path' => $relative,
            'finished_at' => now(),
        ]);
    }
}
