<?php

namespace App\Http\Controllers;

use App\Http\Requests\VoterFilterRequest;
use App\Http\Requests\VoterInlineUpdateRequest;
use App\Http\Requests\VoterStoreRequest;
use App\Http\Requests\VoterUpdateRequest;
use App\Models\ActivityLog;
use App\Models\Area;
use App\Models\Candidate;
use App\Models\SourceFile;
use App\Models\Setting;
use App\Models\Voter;
use App\Queries\VoterFilters;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\StreamedResponse;

class VoterController extends Controller
{
    public function index(VoterFilterRequest $request)
    {
        $filters = $this->normalizeFilters($request->validated());
        $perPage = (int) ($filters['per_page'] ?? 300);
        if (! in_array($perPage, [300, 500, 1000], true)) {
            $perPage = 300;
        }
        $voters = $this->filteredQuery($filters)
            ->paginate($perPage)
            ->withQueryString();

        $areas = Area::orderBy('area_name_bn')->get();
        $candidates = Candidate::where('is_active', true)->orderBy('name_bn')->get();
        $sourceFiles = SourceFile::with('area')->latest()->get();
        $professions = Voter::select('profession_bn')
            ->whereNotNull('profession_bn')
            ->distinct()
            ->orderBy('profession_bn')
            ->pluck('profession_bn');

        return view('admin.voters.index', compact('voters', 'filters', 'areas', 'candidates', 'sourceFiles', 'professions'));
    }

    public function show(Voter $voter)
    {
        $voter->load(['area', 'sourceFile']);

        return view('admin.voters.show', compact('voter'));
    }

    public function create(Request $request)
    {
        $sourceFileId = $request->integer('source_file_id');
        $sourceFile = $sourceFileId ? SourceFile::with('area')->findOrFail($sourceFileId) : null;
        $areas = Area::orderBy('area_name_bn')->get();
        $sourceFiles = SourceFile::with('area')->latest()->get();

        $serialBn = $this->normalizeSerialNoPdfBn($request->input('serial_no_pdf_bn'));
        $genderType = $sourceFile?->area?->gender_type;
        $defaultGender = $genderType === 'male' ? 'male' : ($genderType === 'female' ? 'female' : 'unknown');

        $voter = new Voter([
            'serial_no_pdf_bn' => $serialBn,
            'serial_no' => $serialBn ? (int) $this->bnToEnDigits($serialBn) : null,
            'source_file_id' => $sourceFile?->id,
            'area_id' => $sourceFile?->area_id ?? $request->integer('area_id'),
            'gender' => $defaultGender,
        ]);

        return view('admin.voters.create', compact('voter', 'areas', 'sourceFiles', 'sourceFile'));
    }

    public function store(VoterStoreRequest $request)
    {
        $data = $request->validated();
        $data = array_map(fn ($value) => $value === '' ? null : $value, $data);
        $shiftLater = $request->boolean('shift_later_serials');

        $sourceFile = null;
        if (! empty($data['source_file_id'])) {
            $sourceFile = SourceFile::with('area')->findOrFail($data['source_file_id']);
            $data['area_id'] = $sourceFile->area_id;
        }

        if (empty($data['area_id'])) {
            return back()->withErrors(['area_id' => 'এলাকা নির্বাচন করুন।'])->withInput();
        }

        $serialBn = $this->normalizeSerialNoPdfBn($data['serial_no_pdf_bn'] ?? null);
        if (! $serialBn) {
            return back()->withErrors(['serial_no_pdf_bn' => 'ক্রমিক নম্বর দিন।'])->withInput();
        }
        $data['serial_no_pdf_bn'] = $serialBn;
        $data['serial_no'] = (int) $this->bnToEnDigits($serialBn);

        $duplicateExists = false;
        if (! empty($data['source_file_id'])) {
            $duplicateExists = Voter::where('source_file_id', $data['source_file_id'])
                ->where('serial_no', $data['serial_no'])
                ->exists();
            if ($duplicateExists && ! $shiftLater) {
                return back()->withErrors(['serial_no_pdf_bn' => 'এই ক্রমিক নম্বর ইতিমধ্যে আছে।'])->withInput();
            }
        }

        $voterNo = $this->normalizeVoterNo($data['voter_no'] ?? null);
        if ($voterNo && ! $this->isValidVoterNo($voterNo)) {
            return back()->withErrors(['voter_no' => 'ভোটার নম্বর সঠিক নয়।'])->withInput();
        }
        $data['voter_no'] = $voterNo ?? '';

        $dobBn = $this->normalizeDateOfBirthBn($data['date_of_birth_bn'] ?? null);
        if ($dobBn && ! $this->parseDate($dobBn)) {
            return back()->withErrors(['date_of_birth_bn' => 'জন্মতারিখ সঠিক নয়।'])->withInput();
        }
        $data['date_of_birth_bn'] = $dobBn;
        $data['date_of_birth'] = $this->parseDate($dobBn);

        $data['gender'] = $data['gender'] ?? ($sourceFile?->area?->gender_type === 'male'
            ? 'male'
            : ($sourceFile?->area?->gender_type === 'female' ? 'female' : 'unknown'));

        $voter = DB::transaction(function () use ($data, $duplicateExists, $shiftLater) {
            if ($duplicateExists && $shiftLater && ! empty($data['source_file_id'])) {
                $toShift = Voter::where('source_file_id', $data['source_file_id'])
                    ->whereNotNull('serial_no')
                    ->where('serial_no', '>=', $data['serial_no'])
                    ->orderBy('serial_no', 'desc')
                    ->lockForUpdate()
                    ->get();

                foreach ($toShift as $existing) {
                    $newSerial = (int) $existing->serial_no + 1;
                    $existing->update([
                        'serial_no' => $newSerial,
                        'serial_no_pdf_bn' => $this->formatSerialNoBn($newSerial),
                    ]);
                }
            }

            return Voter::create($data);
        });

        $actor = $request->user();
        ActivityLog::create([
            'actor_id' => $actor?->id,
            'actor_type' => $actor ? $actor::class : null,
            'action' => 'voter.created',
            'subject_id' => $voter->id,
            'subject_type' => Voter::class,
            'changes' => ['created' => true],
            'meta' => [
                'serial_no_pdf_bn' => $data['serial_no_pdf_bn'],
                'source_file_id' => $data['source_file_id'] ?? null,
            ],
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        $redirectParams = array_filter([
            'source_file_id' => $data['source_file_id'] ?? null,
        ], fn ($value) => $value !== null && $value !== '');

        return redirect()
            ->route('admin.voters.index', $redirectParams)
            ->with('status', 'ভোটার যোগ করা হয়েছে।');
    }

    public function edit(Voter $voter)
    {
        return view('admin.voters.edit', compact('voter'));
    }

    public function update(VoterUpdateRequest $request, Voter $voter)
    {
        $data = $request->validated();
        $data = array_map(fn ($value) => $value === '' ? null : $value, $data);
        if (array_key_exists('serial_no_pdf_bn', $data)) {
            $serialBn = $this->normalizeSerialNoPdfBn($data['serial_no_pdf_bn'] ?? null);
            $data['serial_no_pdf_bn'] = $serialBn;
            $data['serial_no'] = $serialBn ? (int) $this->bnToEnDigits($serialBn) : null;
        }
        if (array_key_exists('voter_no', $data)) {
            $voterNo = $this->normalizeVoterNo($data['voter_no'] ?? null);
            if ($voterNo && ! $this->isValidVoterNo($voterNo)) {
                return back()->withErrors(['voter_no' => 'ভোটার নম্বর সঠিক নয়।'])->withInput();
            }
            $data['voter_no'] = $voterNo ?? '';
        }
        $changes = $this->buildChangeSet($voter, $data);

        $voter->update($data);

        if ($changes) {
            $actor = $request->user();
            ActivityLog::create([
                'actor_id' => $actor?->id,
                'actor_type' => $actor ? $actor::class : null,
                'action' => 'voter.updated',
                'subject_id' => $voter->id,
                'subject_type' => Voter::class,
                'changes' => $changes,
                'meta' => ['voter_no' => $voter->voter_no],
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);
        }

        return redirect()
            ->route('admin.voters.show', $voter)
            ->with('status', 'Voter updated successfully.');
    }

    public function inlineUpdate(VoterInlineUpdateRequest $request, Voter $voter)
    {
        $field = (string) $request->input('field');
        $value = $request->input('value');
        $value = is_string($value) ? trim($value) : $value;

        $updates = [];
        $display = '';

        if ($field === 'area_id') {
            $areaId = (int) $value;
            $area = Area::findOrFail($areaId);
            $updates['area_id'] = $areaId;
            $display = $area->area_name_bn;
        } elseif ($field === 'serial_no_pdf_bn') {
            $serialBn = $this->normalizeSerialNoPdfBn($value);
            $updates['serial_no_pdf_bn'] = $serialBn;
            $updates['serial_no'] = $serialBn ? (int) $this->bnToEnDigits($serialBn) : null;
            $display = $serialBn ?? '';
        } elseif ($field === 'date_of_birth_bn') {
            $dobBn = $this->normalizeDateOfBirthBn($value);
            if ($dobBn && ! $this->parseDate($dobBn)) {
                return response()->json(['message' => 'জন্মতারিখ সঠিক নয়।'], 422);
            }
            $updates['date_of_birth_bn'] = $dobBn;
            $updates['date_of_birth'] = $this->parseDate($dobBn);
            $display = $dobBn ?? '';
        } elseif ($field === 'voter_no') {
            $voterNo = $this->normalizeVoterNo($value);
            if ($voterNo && ! $this->isValidVoterNo($voterNo)) {
                return response()->json(['message' => 'ভোটার নম্বর সঠিক নয়।'], 422);
            }
            $updates['voter_no'] = $voterNo ?? '';
            $display = $updates['voter_no'];
        } else {
            $updates[$field] = $value === '' ? null : $value;
            $display = $updates[$field] ?? '';
        }

        $changes = $this->buildChangeSet($voter, $updates);
        $voter->update($updates);

        if ($changes) {
            $actor = $request->user();
            ActivityLog::create([
                'actor_id' => $actor?->id,
                'actor_type' => $actor ? $actor::class : null,
                'action' => 'voter.inline_updated',
                'subject_id' => $voter->id,
                'subject_type' => Voter::class,
                'changes' => $changes,
                'meta' => ['voter_no' => $voter->voter_no],
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);
        }

        return response()->json([
            'ok' => true,
            'field' => $field,
            'value' => $updates[$field] ?? $value,
            'display' => $display,
        ]);
    }

    public function print(VoterFilterRequest $request)
    {
        $filters = $this->normalizeFilters($request->validated());
        $voters = $this->filteredQuery($filters)->limit(1000)->get();
        $candidateId = $request->integer('candidate_id');
        $candidate = $candidateId ? Candidate::findOrFail($candidateId) : null;
        $slipHeader = Setting::getValue('slip_header_bn');
        $slipFooter = Setting::getValue('slip_footer_bn');

        return view('admin.voters.print', compact('voters', 'filters', 'candidate', 'slipHeader', 'slipFooter'));
    }

    public function exportPdf(VoterFilterRequest $request)
    {
        $filters = $this->normalizeFilters($request->validated());
        $voters = $this->filteredQuery($filters)->limit(1000)->get();

        $pdf = Pdf::loadView('admin.voters.pdf', [
            'voters' => $voters,
            'filters' => $filters,
        ])->setPaper('a4', 'portrait');

        return $pdf->download('voters.pdf');
    }

    public function exportCsv(VoterFilterRequest $request): StreamedResponse
    {
        $filters = $this->normalizeFilters($request->validated());
        $query = $this->filteredQuery($filters);

        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="voters.csv"',
        ];

        $callback = function () use ($query) {
            $handle = fopen('php://output', 'w');
            fwrite($handle, chr(239).chr(187).chr(191)); // UTF-8 BOM for Bangla

            fputcsv($handle, ['serial_no', 'serial_no_pdf_bn', 'name_bn', 'voter_no', 'father_name_bn', 'mother_name_bn', 'profession_bn', 'date_of_birth', 'address_bn', 'area']);

            $query->orderBy('serial_no')->orderBy('id')->chunk(500, function ($chunk) use ($handle) {
                foreach ($chunk as $voter) {
                    fputcsv($handle, [
                        $voter->serial_no,
                        $voter->serial_no_pdf_bn ?? $voter->serial_no,
                        $voter->name_bn,
                        $voter->voter_no,
                        $voter->father_name_bn,
                        $voter->mother_name_bn,
                        $voter->profession_bn,
                        optional($voter->date_of_birth)->format('Y-m-d'),
                        $voter->address_bn,
                        $voter->area->area_name_bn ?? '',
                    ]);
                }
            });

            fclose($handle);
        };

        return response()->streamDownload($callback, 'voters.csv', $headers);
    }

    private function filteredQuery(array $filters): Builder
    {
        $filters = $this->normalizeFilters($filters);

        $query = Voter::with(['area', 'sourceFile'])
            ->orderByRaw('serial_no IS NULL')
            ->orderBy('serial_no')
            ->orderBy('id');

        return VoterFilters::apply($query, $filters);
    }

    private function normalizeFilters(array $filters): array
    {
        if (! empty($filters['min_age']) && ! empty($filters['max_age']) && $filters['min_age'] > $filters['max_age']) {
            [$filters['min_age'], $filters['max_age']] = [$filters['max_age'], $filters['min_age']];
        }

        return $filters;
    }

    private function buildChangeSet(Voter $voter, array $data): array
    {
        $changes = [];

        foreach ($data as $field => $value) {
            $original = $this->normalizeChangeValue($voter->getOriginal($field));
            $updated = $this->normalizeChangeValue($value);

            if ($original !== $updated) {
                $changes[$field] = [
                    'from' => $original,
                    'to' => $updated,
                ];
            }
        }

        return $changes;
    }

    private function normalizeChangeValue($value)
    {
        if ($value instanceof \DateTimeInterface) {
            return $value->format('Y-m-d');
        }

        return $value;
    }

    private function normalizeSerialNoPdfBn(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim($value);
        if ($value === '') {
            return null;
        }

        $value = $this->normalizeBnDigits($value);
        $value = preg_replace('/[^০-৯]/u', '', $value);
        if ($value === '') {
            return null;
        }

        $length = mb_strlen($value);
        if ($length < 3) {
            $value = str_repeat('০', 3 - $length).$value;
        }

        return $value;
    }

    private function formatSerialNoBn(int $serial): string
    {
        $serialStr = (string) $serial;
        if (strlen($serialStr) < 3) {
            $serialStr = str_pad($serialStr, 3, '0', STR_PAD_LEFT);
        }

        return $this->normalizeBnDigits($serialStr);
    }

    private function normalizeDateOfBirthBn(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }

        $value = $this->normalizeBnDigits($value);
        $value = str_replace('-', '/', $value);
        $value = preg_replace('/[^০-৯\/]/u', '', $value);

        return $value === '' ? null : $value;
    }

    private function normalizeVoterNo(?string $value): ?string
    {
        if ($value === null) {
            return null;
        }

        $value = trim((string) $value);
        if ($value === '') {
            return null;
        }

        $value = $this->normalizeBnDigits($value);
        $value = preg_replace('/[^০-৯]/u', '', $value);

        return $value === '' ? null : $value;
    }

    private function parseDate(?string $dateString): ?string
    {
        if (! $dateString) {
            return null;
        }

        $converted = $this->bnToEnDigits(trim($dateString));
        foreach (['d/m/Y', 'd-m-Y', 'Y-m-d'] as $format) {
            try {
                return Carbon::createFromFormat($format, $converted)->format('Y-m-d');
            } catch (\Throwable $e) {
                continue;
            }
        }

        return null;
    }

    private function isValidVoterNo(?string $value): bool
    {
        if (! $value) {
            return false;
        }

        $digits = $this->normalizeBnDigits($value);
        $digits = preg_replace('/[^০-৯]/u', '', $digits);
        $length = mb_strlen($digits);

        return $length >= 10 && $length <= 17;
    }

    private function bnToEnDigits(string $value): string
    {
        return strtr($value, [
            '০' => '0', '১' => '1', '২' => '2', '৩' => '3', '৪' => '4',
            '৫' => '5', '৬' => '6', '৭' => '7', '৮' => '8', '৯' => '9',
        ]);
    }

    private function normalizeBnDigits(string $value): string
    {
        return strtr($value, [
            '0' => '০', '1' => '১', '2' => '২', '3' => '৩', '4' => '৪',
            '5' => '৫', '6' => '৬', '7' => '৭', '8' => '৮', '9' => '৯',
        ]);
    }
}
