<?php

namespace App\Http\Controllers;

use App\Models\Area;
use App\Models\SourceFile;
use App\Jobs\ImportSourceFilePage;
use App\Services\VoterPdfImportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class SourceFileController extends Controller
{
    public function __construct(private readonly VoterPdfImportService $importService)
    {
    }

    public function index()
    {
        $sourceFiles = SourceFile::with('area')
            ->withCount('voters')
            ->where('stored_path', 'like', 'voter_pdfs/%')
            ->latest()
            ->paginate(15);

        return view('admin.source_files.index', compact('sourceFiles'));
    }

    public function create()
    {
        $areas = Area::orderBy('area_name_bn')->get();

        return view('admin.source_files.create', compact('areas'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'area_id' => ['nullable', 'exists:areas,id'],
            'file' => ['required', 'file', 'mimes:pdf', 'max:102400'],
            'notes' => ['nullable', 'string'],
            'voter_gender' => ['nullable', 'in:male,female,unknown'],
            'new_area.district' => ['nullable', 'string', 'max:255'],
            'new_area.upazila' => ['nullable', 'string', 'max:255'],
            'new_area.city_corporation' => ['nullable', 'string', 'max:255'],
            'new_area.ward_no' => ['nullable', 'string', 'max:50'],
            'new_area.area_code' => ['nullable', 'string', 'max:50'],
            'new_area.area_name_bn' => ['nullable', 'string', 'max:255'],
            'new_area.center_name_bn' => ['nullable', 'string', 'max:255'],
            'new_area.gender_type' => ['nullable', 'in:male,female,mixed'],
            'new_area.raw_header_text' => ['nullable', 'string'],
        ], [], [
            'area_id' => 'এলাকা',
            'file' => 'পিডিএফ ফাইল',
            'notes' => 'নোট',
            'voter_gender' => 'Voter gender',
            'new_area.area_name_bn' => 'নতুন এলাকা নাম',
        ]);

        $area = null;
        if ($request->filled('area_id')) {
            $area = Area::findOrFail($request->integer('area_id'));
        }

        $newAreaData = $request->input('new_area', []);
        if (! $area && $this->hasNewAreaData($newAreaData)) {
            if (empty($newAreaData['area_name_bn'])) {
                return back()->withErrors(['new_area.area_name_bn' => 'নতুন এলাকার নাম প্রয়োজন।'])->withInput();
            }

            $newAreaData['gender_type'] = $newAreaData['gender_type'] ?? 'mixed';
            $area = Area::create($newAreaData);
        }

        if (! $area) {
            return back()->withErrors(['area_id' => 'এলাকা নির্বাচন করুন অথবা নতুন এলাকা তৈরি করুন।'])->withInput();
        }

        $file = $validated['file'];
        $path = $file->store('voter_pdfs');

        $sourceFile = SourceFile::create([
            'area_id' => $area->id,
            'original_filename' => $file->getClientOriginalName(),
            'stored_path' => $path,
            'mime_type' => $file->getClientMimeType(),
            'file_size' => $file->getSize(),
            'notes' => $request->input('notes'),
            'import_status' => 'pending',
            'imported_pages' => 0,
            'total_pages' => null,
            'import_options' => [
                'default_voter_gender' => $validated['voter_gender'] ?? null,
            ],
        ]);

        return redirect()
            ->route('admin.source-files.import', $sourceFile)
            ->with('status', 'পিডিএফ আপলোড সম্পন্ন হয়েছে। এখন ইম্পোর্ট শুরু করুন।');
    }

    public function show(SourceFile $sourceFile)
    {
        return redirect()->route('admin.source-files.index');
    }

    public function edit(SourceFile $sourceFile)
    {
        return redirect()->route('admin.source-files.index');
    }

    public function update(Request $request, SourceFile $sourceFile)
    {
        return redirect()->route('admin.source-files.index');
    }

    public function importForm(SourceFile $sourceFile)
    {
        $pageCount = $this->importService->getPdfTotalPages($sourceFile);

        return view('admin.source_files.import', [
            'sourceFile' => $sourceFile->load('area'),
            'pageCount' => $pageCount,
            'importOptions' => $sourceFile->import_options ?? [],
        ]);
    }

    public function startImport(Request $request, SourceFile $sourceFile)
    {
        $validated = $request->validate([
            'page_mode' => ['required', 'in:all,single,range'],
            'page_number' => ['nullable', 'integer', 'min:1'],
            'page_from' => ['nullable', 'integer', 'min:1'],
            'page_to' => ['nullable', 'integer', 'min:1'],
            'voter_gender' => ['nullable', 'in:male,female,unknown'],
        ], [], [
            'page_mode' => 'পেজ ধরন',
            'page_number' => 'পেজ নম্বর',
            'page_from' => 'শুরু পেজ',
            'page_to' => 'শেষ পেজ',
        ]);

        $pageCount = $this->importService->getPdfTotalPages($sourceFile);
        $pageMode = $validated['page_mode'];
        $pages = [];

        if ($pageMode === 'all') {
            if (! $pageCount) {
                return back()->withErrors(['page_mode' => 'পেজ সংখ্যা নির্ধারণ করা যায়নি, অনুগ্রহ করে রেঞ্জ দিন।'])->withInput();
            }
            $pages = range(1, $pageCount);
        } elseif ($pageMode === 'single') {
            if (empty($validated['page_number'])) {
                return back()->withErrors(['page_number' => 'পেজ নম্বর দিন।'])->withInput();
            }
            $pages = [(int) $validated['page_number']];
        } else {
            if (empty($validated['page_from']) || empty($validated['page_to'])) {
                return back()->withErrors(['page_from' => 'শুরু ও শেষ পেজ দিন।'])->withInput();
            }
            $from = (int) $validated['page_from'];
            $to = (int) $validated['page_to'];
            if ($from > $to) {
                return back()->withErrors(['page_from' => 'শুরু পেজ শেষ পেজের চেয়ে বড় হতে পারে না।'])->withInput();
            }
            $pages = range($from, $to);
        }

        if ($pageCount) {
            $invalid = array_filter($pages, fn ($page) => $page < 1 || $page > $pageCount);
            if ($invalid) {
                return back()->withErrors(['page_number' => 'পেজ নম্বর সীমার বাইরে আছে।'])->withInput();
            }
        }

        $voterGender = $validated['voter_gender'] ?? ($sourceFile->import_options['default_voter_gender'] ?? null);
        $importToken = (string) Str::uuid();

        $importOptions = array_merge($sourceFile->import_options ?? [], [
            'page_mode' => $pageMode,
            'page_number' => $validated['page_number'] ?? null,
            'page_from' => $validated['page_from'] ?? null,
            'page_to' => $validated['page_to'] ?? null,
            'pages' => $pages,
            'pdf_total_pages' => $pageCount,
            'voter_gender' => $voterGender,
            'import_token' => $importToken,
        ]);

        $sourceFile->update([
            'import_status' => 'queued',
            'imported_pages' => 0,
            'total_pages' => count($pages),
            'import_started_at' => now(),
            'import_finished_at' => null,
            'import_options' => $importOptions,
        ]);

        foreach ($pages as $page) {
            ImportSourceFilePage::dispatch($sourceFile->id, $page, $voterGender, $importToken);
        }

        return redirect()
            ->route('admin.source-files.import', $sourceFile)
            ->with('status', 'ইম্পোর্ট শুরু হয়েছে। প্রোগ্রেস দেখুন।');
    }

    public function stopImport(SourceFile $sourceFile)
    {
        $importOptions = $sourceFile->import_options ?? [];
        $importOptions['import_token'] = (string) Str::uuid();

        $sourceFile->update([
            'import_status' => 'stopped',
            'import_finished_at' => now(),
            'import_options' => $importOptions,
        ]);

        return redirect()
            ->route('admin.source-files.import', $sourceFile)
            ->with('status', 'ইম্পোর্ট বন্ধ হয়েছে।');
    }

    public function restartImport(SourceFile $sourceFile)
    {
        $importOptions = $sourceFile->import_options ?? [];
        $pages = $importOptions['pages'] ?? [];
        $pageCount = $this->importService->getPdfTotalPages($sourceFile);

        if (empty($pages)) {
            $pageMode = $importOptions['page_mode'] ?? 'all';
            if ($pageMode === 'all' && $pageCount) {
                $pages = range(1, $pageCount);
            } elseif ($pageMode === 'single' && ! empty($importOptions['page_number'])) {
                $pages = [(int) $importOptions['page_number']];
            } elseif ($pageMode === 'range' && ! empty($importOptions['page_from']) && ! empty($importOptions['page_to'])) {
                $from = (int) $importOptions['page_from'];
                $to = (int) $importOptions['page_to'];
                if ($from <= $to) {
                    $pages = range($from, $to);
                }
            }
        }

        if (empty($pages)) {
            return back()->withErrors(['page_mode' => 'ইম্পোর্ট সেটিংস পাওয়া যায়নি। নতুন করে ইম্পোর্ট শুরু করুন।'])->withInput();
        }

        $voterGender = $importOptions['voter_gender'] ?? $importOptions['default_voter_gender'] ?? null;
        $importToken = (string) Str::uuid();

        $importOptions = array_merge($importOptions, [
            'pages' => $pages,
            'pdf_total_pages' => $pageCount,
            'voter_gender' => $voterGender,
            'import_token' => $importToken,
        ]);

        $sourceFile->update([
            'import_status' => 'queued',
            'imported_pages' => 0,
            'total_pages' => count($pages),
            'import_started_at' => now(),
            'import_finished_at' => null,
            'import_options' => $importOptions,
        ]);

        foreach ($pages as $page) {
            ImportSourceFilePage::dispatch($sourceFile->id, $page, $voterGender, $importToken);
        }

        return redirect()
            ->route('admin.source-files.import', $sourceFile)
            ->with('status', 'ইম্পোর্ট আবার শুরু হয়েছে।');
    }

    public function importStatus(SourceFile $sourceFile)
    {
        $total = $sourceFile->total_pages ?? 0;
        $imported = $sourceFile->imported_pages ?? 0;
        $percentage = $total > 0 ? (int) round(($imported / $total) * 100) : 0;

        return response()->json([
            'status' => $sourceFile->import_status,
            'imported_pages' => $imported,
            'total_pages' => $total,
            'percentage' => $percentage,
            'started_at' => optional($sourceFile->import_started_at)->toDateTimeString(),
            'finished_at' => optional($sourceFile->import_finished_at)->toDateTimeString(),
        ]);
    }

    public function destroy(SourceFile $sourceFile)
    {
        // Delete linked voters, then delete file record and stored PDF
        $sourceFile->voters()->delete();

        if ($sourceFile->stored_path) {
            Storage::delete($sourceFile->stored_path);
        }

        $sourceFile->delete();

        return redirect()
            ->route('admin.source-files.index')
            ->with('status', 'ফাইল ও সংশ্লিষ্ট ভোটার ডেটা মুছে ফেলা হয়েছে।');
    }

    private function hasNewAreaData(array $data): bool
    {
        return collect($data)->filter(fn ($value) => filled($value))->isNotEmpty();
    }
}
