<?php

namespace App\Http\Controllers;

use App\Models\Candidate;
use App\Models\Setting;
use App\Models\SourceFile;
use App\Models\Voter;
use Illuminate\Http\Request;

class SlipPrintController extends Controller
{
    private const PER_PAGE = 10;

    public function index()
    {
        $sourceFiles = SourceFile::with('area')->latest()->get();
        $candidates = Candidate::where('is_active', true)->orderBy('name_bn')->get();

        return view('admin.slip_prints.index', compact('sourceFiles', 'candidates'));
    }

    public function preview(Request $request)
    {
        $validated = $request->validate([
            'source_file_id' => ['required', 'integer', 'exists:source_files,id'],
            'candidate_id' => ['required', 'integer', 'exists:candidates,id'],
            'pages' => ['nullable', 'integer', 'min:1', 'required_without:total_voters'],
            'total_voters' => ['nullable', 'integer', 'min:1', 'required_without:pages'],
        ], [
            'pages.required_without' => 'Please provide pages or total voters.',
            'total_voters.required_without' => 'Please provide pages or total voters.',
        ]);

        $pages = $validated['pages'] ?? null;
        $totalVoters = $validated['total_voters'] ?? null;

        $limit = $totalVoters ?: ($pages * self::PER_PAGE);
        $computedPages = (int) ceil($limit / self::PER_PAGE);

        $sourceFile = SourceFile::with('area')->findOrFail($validated['source_file_id']);
        $candidate = Candidate::findOrFail($validated['candidate_id']);
        $slipHeader = Setting::getValue('slip_header_bn');
        $slipFooter = Setting::getValue('slip_footer_bn');

        $voters = Voter::with('area')
            ->where('source_file_id', $sourceFile->id)
            ->orderByRaw($this->serialOrderSql())
            ->limit($limit)
            ->get();

        return view('admin.slip_prints.print', [
            'voters' => $voters,
            'candidate' => $candidate,
            'sourceFile' => $sourceFile,
            'perPage' => self::PER_PAGE,
            'pages' => $computedPages,
            'slipHeader' => $slipHeader,
            'slipFooter' => $slipFooter,
        ]);
    }

    private function serialOrderSql(): string
    {
        $expr = 'serial_no_pdf_bn';
        $map = [
            "\u{09E6}" => '0',
            "\u{09E7}" => '1',
            "\u{09E8}" => '2',
            "\u{09E9}" => '3',
            "\u{09EA}" => '4',
            "\u{09EB}" => '5',
            "\u{09EC}" => '6',
            "\u{09ED}" => '7',
            "\u{09EE}" => '8',
            "\u{09EF}" => '9',
        ];

        foreach ($map as $bn => $en) {
            $expr = "REPLACE($expr, '{$bn}', '{$en}')";
        }

        return "CASE WHEN serial_no_pdf_bn IS NULL OR serial_no_pdf_bn = '' THEN serial_no ELSE CAST($expr AS UNSIGNED) END";
    }
}
