<?php

namespace App\Http\Controllers;

use App\Models\SourceFile;
use App\Services\PdfSlipPrintService;
use App\Services\VoterPdfImportService;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class PdfSlipPrintController extends Controller
{
    public function index()
    {
        $sourceFiles = SourceFile::with('area')->latest()->get();

        return view('admin.pdf_slip_prints.index', compact('sourceFiles'));
    }

    public function preview(Request $request, PdfSlipPrintService $service)
    {
        $validated = $request->validate([
            'source_file_id' => ['required', 'integer', 'exists:source_files,id'],
            'page_from' => ['required', 'integer', 'min:1'],
            'page_to' => ['required', 'integer', 'min:1', 'gte:page_from'],
            'dpi' => ['nullable', 'integer', 'min:120', 'max:600'],
            'cols' => ['nullable', 'integer', 'min:1', 'max:4'],
            'rows' => ['nullable', 'integer', 'min:1', 'max:10'],
            'margin_top' => ['nullable', 'string', 'max:10'],
            'margin_bottom' => ['nullable', 'string', 'max:10'],
            'margin_left' => ['nullable', 'string', 'max:10'],
            'margin_right' => ['nullable', 'string', 'max:10'],
            'inset' => ['nullable', 'numeric', 'min:0', 'max:0.1'],
        ], [
            'page_to.gte' => 'শেষ পেজ শুরু পেজের চেয়ে ছোট হতে পারবে না।',
        ]);

        $sourceFile = SourceFile::with('area')->findOrFail($validated['source_file_id']);
        $options = array_intersect_key($validated, array_flip([
            'dpi',
            'cols',
            'rows',
            'margin_top',
            'margin_bottom',
            'margin_left',
            'margin_right',
            'inset',
            'box_height_scale',
            'box_pad',
        ]));
        // Force fixed 2x5 layout (10 slips per page)
        $options['cols'] = 2;
        $options['rows'] = 5;

        try {
            $result = $service->build($sourceFile, (int) $validated['page_from'], (int) $validated['page_to'], $options);
        } catch (\Throwable $e) {
            return back()->withErrors(['source_file_id' => $e->getMessage()])->withInput();
        }

        return view('admin.pdf_slip_prints.print', [
            'sourceFile' => $sourceFile,
            'token' => $result['token'],
            'pages' => $result['pages'],
            'options' => $result['options'],
            'flatFiles' => $this->flattenFiles($result['pages']),
        ]);
    }

    public function cropStart(Request $request, PdfSlipPrintService $service, VoterPdfImportService $importService)
    {
        $validated = $request->validate([
            'source_file_id' => ['required', 'integer', 'exists:source_files,id'],
            'page_from' => ['nullable', 'integer', 'min:1'],
            'page_to' => ['nullable', 'integer', 'min:1'],
            'dpi' => ['nullable', 'integer', 'min:120', 'max:600'],
            'margin_top' => ['nullable', 'string', 'max:10'],
            'margin_bottom' => ['nullable', 'string', 'max:10'],
            'margin_left' => ['nullable', 'string', 'max:10'],
            'margin_right' => ['nullable', 'string', 'max:10'],
            'inset' => ['nullable', 'numeric', 'min:0', 'max:0.1'],
            'box_height_scale' => ['nullable', 'numeric', 'min:0.7', 'max:1.3'],
            'box_pad' => ['nullable', 'integer', 'min:0', 'max:40'],
            'detect_lines' => ['nullable', 'boolean'],
        ]);

        $sourceFile = SourceFile::with('area')->findOrFail($validated['source_file_id']);
        $pageCount = $importService->getPdfTotalPages($sourceFile);
        if (! $pageCount) {
            return response()->json(['message' => 'PDF page count not found.'], 422);
        }

        $pageFrom = (int) ($validated['page_from'] ?? 1);
        $pageTo = (int) ($validated['page_to'] ?? $pageCount);
        $pageFrom = max(1, min($pageFrom, $pageCount));
        $pageTo = max($pageFrom, min($pageTo, $pageCount));

        $options = array_intersect_key($validated, array_flip([
            'dpi',
            'margin_top',
            'margin_bottom',
            'margin_left',
            'margin_right',
            'inset',
            'box_height_scale',
            'box_pad',
            'detect_lines',
        ]));
        // Crop should follow native 3-column layout, rows auto (5-6) by page.
        $options['cols'] = 3;
        $options['rows'] = 0;
        $options['auto_rows_min'] = 5;
        $options['auto_rows_max'] = 6;
        if (! array_key_exists('detect_lines', $options)) {
            $options['detect_lines'] = true;
        }

        try {
            $status = $service->startCropSession($sourceFile, $pageFrom, $pageTo, $options);
        } catch (\Throwable $e) {
            return response()->json(['message' => $e->getMessage()], 422);
        }

        return response()->json($status);
    }

    public function cropPage(Request $request, PdfSlipPrintService $service)
    {
        $validated = $request->validate([
            'source_file_id' => ['required', 'integer', 'exists:source_files,id'],
            'token' => ['required', 'string'],
            'page' => ['required', 'integer', 'min:1'],
        ]);

        $sourceFile = SourceFile::with('area')->findOrFail($validated['source_file_id']);
        try {
            $status = $service->cropPage($sourceFile, $validated['token'], (int) $validated['page']);
        } catch (\Throwable $e) {
            return response()->json(['message' => $e->getMessage()], 422);
        }

        return response()->json($status);
    }

    public function cropStatus(PdfSlipPrintService $service, SourceFile $sourceFile, string $token)
    {
        $status = $service->getCropStatus($sourceFile, $token);
        if (empty($status)) {
            return response()->json(['message' => 'Crop session not found.'], 404);
        }

        return response()->json($status);
    }

    public function cropPrint(Request $request, PdfSlipPrintService $service)
    {
        $validated = $request->validate([
            'source_file_id' => ['required', 'integer', 'exists:source_files,id'],
            'token' => ['required', 'string'],
            'serial_from' => ['nullable', 'integer', 'min:1'],
            'serial_to' => ['nullable', 'integer', 'min:1'],
        ]);

        $sourceFile = SourceFile::with('area')->findOrFail($validated['source_file_id']);
        $manifest = $service->getCropManifest($sourceFile, $validated['token']);

        $serialFrom = (int) ($validated['serial_from'] ?? 1);
        $serialTo = (int) ($validated['serial_to'] ?? PHP_INT_MAX);
        if ($serialTo < $serialFrom) {
            $serialTo = $serialFrom;
        }

        $filtered = array_values(array_filter($manifest, function ($entry) use ($serialFrom, $serialTo) {
            $serial = (int) ($entry['serial'] ?? 0);
            return $serial >= $serialFrom && $serial <= $serialTo;
        }));

        return view('admin.pdf_slip_prints.print_cropped', [
            'sourceFile' => $sourceFile,
            'token' => $validated['token'],
            'options' => [
                'cols' => 2,
                'rows' => 5,
            ],
            'flatFiles' => array_map(function ($entry) {
                return [
                    'serial' => $entry['serial'] ?? null,
                    'file' => $entry['file'] ?? null,
                ];
            }, $filtered),
        ]);
    }

    public function cropImage(SourceFile $sourceFile, string $token, string $file)
    {
        $file = basename($file);
        $path = storage_path('app/pdf_slip_crops/sf-'.$sourceFile->id.'/'.$token.'/'.$file);
        if (! file_exists($path)) {
            abort(Response::HTTP_NOT_FOUND);
        }

        return response()->file($path, [
            'Content-Type' => 'image/png',
            'Cache-Control' => 'private, max-age=3600',
        ]);
    }

    private function flattenFiles(array $pages): array
    {
        ksort($pages);
        $flat = [];
        foreach ($pages as $page => $files) {
            usort($files, function ($a, $b) {
                $aIndex = $this->extractBlockIndex($a);
                $bIndex = $this->extractBlockIndex($b);
                return $aIndex <=> $bIndex;
            });
            foreach ($files as $file) {
                $flat[] = [
                    'page' => $page,
                    'file' => $file,
                ];
            }
        }

        return $flat;
    }

    private function extractBlockIndex(string $file): int
    {
        if (preg_match('/block-(\d+)\.png$/', $file, $matches)) {
            return (int) $matches[1];
        }
        return 0;
    }

    public function image(string $token, string $file)
    {
        $file = basename($file);
        $path = storage_path('app/pdf_slips/'.$token.'/'.$file);
        if (! file_exists($path)) {
            abort(Response::HTTP_NOT_FOUND);
        }

        return response()->file($path, [
            'Content-Type' => 'image/png',
            'Cache-Control' => 'private, max-age=3600',
        ]);
    }
}
