<?php

namespace App\Http\Controllers;

use App\Models\Area;
use App\Models\SourceFile;
use App\Services\VoterCsvImportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CsvFileController extends Controller
{
    public function __construct(private readonly VoterCsvImportService $importService)
    {
    }

    public function index()
    {
        $sourceFiles = SourceFile::with('area')
            ->withCount('voters')
            ->where('stored_path', 'like', 'voter_csvs/%')
            ->latest()
            ->paginate(15);

        return view('admin.csv_files.index', compact('sourceFiles'));
    }

    public function create()
    {
        $areas = Area::orderBy('area_name_bn')->get();

        return view('admin.csv_files.create', compact('areas'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'area_id' => ['nullable', 'exists:areas,id'],
            'file' => ['required', 'file', 'mimes:csv,txt', 'max:20480'],
            'notes' => ['nullable', 'string'],
            'new_area.district' => ['nullable', 'string', 'max:255'],
            'new_area.upazila' => ['nullable', 'string', 'max:255'],
            'new_area.city_corporation' => ['nullable', 'string', 'max:255'],
            'new_area.ward_no' => ['nullable', 'string', 'max:50'],
            'new_area.area_code' => ['nullable', 'string', 'max:50'],
            'new_area.area_name_bn' => ['nullable', 'string', 'max:255'],
            'new_area.center_name_bn' => ['nullable', 'string', 'max:255'],
            'new_area.gender_type' => ['nullable', 'in:male,female,mixed'],
            'new_area.raw_header_text' => ['nullable', 'string'],
        ], [], [
            'area_id' => 'এলাকা',
            'file' => 'CSV ফাইল',
            'notes' => 'নোট',
            'new_area.area_name_bn' => 'নতুন এলাকার নাম',
        ]);

        $area = null;
        if ($request->filled('area_id')) {
            $area = Area::findOrFail($request->integer('area_id'));
        }

        $newAreaData = $request->input('new_area', []);
        if (! $area && $this->hasNewAreaData($newAreaData)) {
            if (empty($newAreaData['area_name_bn'])) {
                return back()->withErrors([
                    'new_area.area_name_bn' => 'নতুন এলাকার নাম দিন।',
                ])->withInput();
            }

            $newAreaData['gender_type'] = $newAreaData['gender_type'] ?? 'mixed';
            $area = Area::create($newAreaData);
        }

        if (! $area) {
            return back()->withErrors([
                'area_id' => 'এলাকা নির্বাচন করুন অথবা নতুন এলাকা তৈরি করুন।',
            ])->withInput();
        }

        $file = $validated['file'];
        $path = $file->store('voter_csvs');

        $sourceFile = SourceFile::create([
            'area_id' => $area->id,
            'original_filename' => $file->getClientOriginalName(),
            'stored_path' => $path,
            'mime_type' => $file->getClientMimeType(),
            'file_size' => $file->getSize(),
            'notes' => $request->input('notes'),
            'parsed_successfully' => false,
            'parsed_at' => null,
            'import_status' => 'pending',
            'imported_pages' => 0,
            'total_pages' => null,
            'import_options' => [
                'file_kind' => 'csv',
            ],
        ]);

        try {
            $summary = $this->importService->import($sourceFile);
            $success = $summary['failed_rows'] === 0 && $summary['imported_rows'] > 0;
            $importOptions = array_merge($sourceFile->import_options ?? [], [
                'file_kind' => 'csv',
                'csv_total_rows' => $summary['total_rows'],
                'csv_imported_rows' => $summary['imported_rows'],
                'csv_failed_rows' => $summary['failed_rows'],
            ]);

            $sourceFile->update([
                'parsed_successfully' => $success,
                'parsed_at' => now(),
                'import_status' => $success ? 'completed' : 'failed',
                'import_started_at' => now(),
                'import_finished_at' => now(),
                'imported_pages' => $summary['imported_rows'],
                'total_pages' => $summary['total_rows'],
                'import_options' => $importOptions,
                'notes' => $summary['failed_rows']
                    ? 'কিছু সারি ইম্পোর্ট হয়নি।'
                    : 'CSV ইম্পোর্ট সম্পন্ন।',
            ]);
        } catch (\Throwable $e) {
            $sourceFile->update([
                'parsed_successfully' => false,
                'parsed_at' => now(),
                'import_status' => 'failed',
                'import_finished_at' => now(),
                'notes' => 'CSV ইম্পোর্ট ব্যর্থ: '.$e->getMessage(),
            ]);

            return redirect()
                ->route('admin.csv-files.index')
                ->with('status', 'CSV ইম্পোর্ট ব্যর্থ হয়েছে।')
                ->with('import_summary', [
                    'total_rows' => 0,
                    'imported_rows' => 0,
                    'failed_rows' => 0,
                    'failed_samples' => [],
                    'error_message' => $e->getMessage(),
                ]);
        }

        return redirect()
            ->route('admin.csv-files.index')
            ->with('status', 'CSV আপলোড ও ইম্পোর্ট সম্পন্ন হয়েছে।')
            ->with('import_summary', $summary);
    }

    public function downloadSample()
    {
        $path = base_path('resources/templates/voters_sample.csv');
        if (! file_exists($path)) {
            abort(404);
        }

        return response()->download($path, 'voters_sample.csv', [
            'Content-Type' => 'text/csv; charset=UTF-8',
        ]);
    }

    public function destroy(SourceFile $sourceFile)
    {
        if (! str_starts_with((string) $sourceFile->stored_path, 'voter_csvs/')) {
            return redirect()
                ->route('admin.csv-files.index')
                ->with('status', 'শুধু CSV ফাইল মুছে ফেলা যাবে।');
        }

        $sourceFile->voters()->delete();

        if ($sourceFile->stored_path) {
            Storage::delete($sourceFile->stored_path);
        }

        $sourceFile->delete();

        return redirect()
            ->route('admin.csv-files.index')
            ->with('status', 'CSV ফাইল ও সংশ্লিষ্ট ভোটার ডেটা মুছে ফেলা হয়েছে।');
    }

    private function hasNewAreaData(array $data): bool
    {
        return collect($data)->filter(fn ($value) => filled($value))->isNotEmpty();
    }
}
