<?php

namespace App\Http\Controllers;

use App\Models\Candidate;
use App\Models\PdfExport;
use App\Models\Setting;
use App\Models\SourceFile;
use App\Services\PdfSlipPrintService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response;

class CandidateSlipPrintController extends Controller
{
    private const PER_PAGE = 10;

    public function index(PdfSlipPrintService $service)
    {
        $sourceFiles = SourceFile::with('area')->latest()->get();
        $candidates = Candidate::where('is_active', true)->orderBy('name_bn')->get();

        return view('admin.candidate_slip_prints.index', compact('sourceFiles', 'candidates'));
    }

    public function preview(Request $request, PdfSlipPrintService $service)
    {
        $payload = $this->validatePayload($request);
        $data = $this->buildSlipData($payload, $service);

        return view('admin.candidate_slip_prints.print', $data);
    }

    public function downloadPdf(Request $request, PdfSlipPrintService $service)
    {
        $payload = $this->validatePayload($request);
        $data = $this->buildSlipData($payload, $service, true);

        $pdf = Pdf::loadView('admin.candidate_slip_prints.pdf', $data)
            ->setPaper('a4')
            ->setOption('isRemoteEnabled', true);

        $filename = 'candidate-slip-'.$data['sourceFile']->id.'-'.now()->format('YmdHis').'.pdf';

        return $pdf->download($filename);
    }

    public function startPdfExport(Request $request)
    {
        $payload = $this->validatePayload($request);
        $sourceFile = SourceFile::findOrFail($payload['source_file_id']);
        $candidate = Candidate::findOrFail($payload['candidate_id']);

        $export = PdfExport::create([
            'source_file_id' => $sourceFile->id,
            'candidate_id' => $candidate->id,
            'serial_from' => (int) ($payload['serial_from'] ?? 1),
            'serial_to' => $payload['serial_to'] ? (int) $payload['serial_to'] : null,
            'status' => 'pending',
            'progress' => 0,
        ]);

        \App\Jobs\GenerateCandidateSlipPdf::dispatch($export->id);

        return response()->json([
            'export_id' => $export->id,
            'status' => $export->status,
        ]);
    }

    public function exportStatus(PdfExport $export)
    {
        return response()->json([
            'id' => $export->id,
            'status' => $export->status,
            'progress' => $export->progress,
            'message' => $export->message,
            'total_slips' => $export->total_slips,
            'processed_slips' => $export->processed_slips,
            'download_url' => $export->status === 'done'
                ? route('admin.candidate-slip-print.pdf.download', $export)
                : null,
        ]);
    }

    public function cancelExport(PdfExport $export)
    {
        if ($export->status === 'done') {
            return response()->json(['status' => $export->status]);
        }

        $export->update([
            'status' => 'canceled',
            'canceled_at' => now(),
        ]);

        return response()->json(['status' => $export->status]);
    }

    public function downloadExport(PdfExport $export)
    {
        if ($export->status !== 'done' || ! $export->file_path) {
            abort(Response::HTTP_NOT_FOUND);
        }

        $path = Storage::disk('local')->path($export->file_path);
        if (! file_exists($path)) {
            abort(Response::HTTP_NOT_FOUND);
        }

        return response()->download($path, basename($path));
    }

    public function image(Candidate $candidate, string $type)
    {
        $type = strtolower($type);
        $path = $type === 'symbol' ? $candidate->symbol_path : $candidate->photo_path;
        if (! $path || ! Storage::disk('public')->exists($path)) {
            abort(Response::HTTP_NOT_FOUND);
        }

        $fullPath = Storage::disk('public')->path($path);

        return response()->file($fullPath, [
            'Content-Type' => 'image/'.pathinfo($fullPath, PATHINFO_EXTENSION),
            'Cache-Control' => 'private, max-age=3600',
        ]);
    }

    private function validatePayload(Request $request): array
    {
        return $request->validate([
            'source_file_id' => ['required', 'integer', 'exists:source_files,id'],
            'candidate_id' => ['required', 'integer', 'exists:candidates,id'],
            'serial_from' => ['nullable', 'integer', 'min:1'],
            'serial_to' => ['nullable', 'integer', 'min:1'],
        ]);
    }

    private function buildSlipData(array $payload, PdfSlipPrintService $service, bool $forPdf = false): array
    {
        $sourceFile = SourceFile::with('area')->findOrFail($payload['source_file_id']);
        $candidate = Candidate::findOrFail($payload['candidate_id']);

        $token = $service->getLatestCropToken($sourceFile);
        if (! $token) {
            abort(422, 'এই সোর্স ফাইলের জন্য ক্রপ ডাটা পাওয়া যায়নি। আগে PDF ক্রপ করুন।');
        }

        $manifest = $service->getCropManifest($sourceFile, $token);
        if (empty($manifest)) {
            abort(422, 'এই সোর্স ফাইলের জন্য ক্রপ ডাটা পাওয়া যায়নি। আগে PDF ক্রপ করুন।');
        }

        $serialFrom = (int) ($payload['serial_from'] ?? 1);
        $serialTo = (int) ($payload['serial_to'] ?? PHP_INT_MAX);
        if ($serialTo < $serialFrom) {
            $serialTo = $serialFrom;
        }

        $filtered = array_values(array_filter($manifest, function ($entry) use ($serialFrom, $serialTo) {
            $serial = (int) ($entry['serial'] ?? 0);
            return $serial >= $serialFrom && $serial <= $serialTo;
        }));

        $slipHeader = Setting::getValue('slip_header_bn');
        $slipFooter = Setting::getValue('slip_footer_bn');
        $area = $sourceFile->area;
        $locationParts = [];
        if (! empty($area?->center_name_bn)) {
            $locationParts[] = 'কেন্দ্র: '.$area->center_name_bn;
        }
        if (! empty($area?->area_name_bn)) {
            $locationParts[] = 'এলাকা: '.$area->area_name_bn;
        }
        if (! empty($area?->ward_no)) {
            $locationParts[] = 'ওয়ার্ড: '.$area->ward_no;
        }
        $locationLine = implode(', ', $locationParts);

        $photoUrl = null;
        if ($candidate->photo_path && Storage::disk('public')->exists($candidate->photo_path)) {
            $photoUrl = route('admin.candidate-slip-print.image', ['candidate' => $candidate->id, 'type' => 'photo']);
        }
        $symbolUrl = null;
        if ($candidate->symbol_path && Storage::disk('public')->exists($candidate->symbol_path)) {
            $symbolUrl = route('admin.candidate-slip-print.image', ['candidate' => $candidate->id, 'type' => 'symbol']);
        }

        if ($forPdf) {
            $photoUrl = $photoUrl ? url($photoUrl) : null;
            $symbolUrl = $symbolUrl ? url($symbolUrl) : null;
        }

        $flatFiles = array_map(function ($entry) use ($sourceFile, $token, $forPdf) {
            $file = $entry['file'] ?? null;
            $imageUrl = $file ? route('admin.pdf-slip-print.crop.image', [
                'sourceFile' => $sourceFile->id,
                'token' => $token,
                'file' => $file,
            ]) : null;
            if ($imageUrl && $forPdf) {
                $imageUrl = url($imageUrl);
            }
            return [
                'serial' => $entry['serial'] ?? null,
                'file' => $file,
                'image_url' => $imageUrl,
            ];
        }, $filtered);

        return [
            'sourceFile' => $sourceFile,
            'candidate' => $candidate,
            'token' => $token,
            'flatFiles' => $flatFiles,
            'perPage' => self::PER_PAGE,
            'slipHeader' => $slipHeader,
            'slipFooter' => $slipFooter,
            'locationLine' => $locationLine,
            'candidatePhotoUrl' => $photoUrl,
            'candidateSymbolUrl' => $symbolUrl,
        ];
    }
}
