<?php

namespace App\Http\Controllers;

use App\Models\Candidate;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

class CandidateController extends Controller
{
    public function index()
    {
        $candidates = Candidate::latest()->paginate(15);

        return view('admin.candidates.index', compact('candidates'));
    }

    public function create()
    {
        $candidate = new Candidate();

        return view('admin.candidates.create', compact('candidate'));
    }

    public function store(Request $request)
    {
        $data = $this->validateCandidate($request);
        $payload = $this->buildPayload($request, $data);

        Candidate::create($payload);

        return redirect()
            ->route('admin.candidates.index')
            ->with('status', 'Candidate created successfully.');
    }

    public function show(Candidate $candidate)
    {
        return view('admin.candidates.show', compact('candidate'));
    }

    public function edit(Candidate $candidate)
    {
        return view('admin.candidates.edit', compact('candidate'));
    }

    public function update(Request $request, Candidate $candidate)
    {
        $data = $this->validateCandidate($request);
        $payload = $this->buildPayload($request, $data, $candidate);

        $candidate->update($payload);

        return redirect()
            ->route('admin.candidates.index')
            ->with('status', 'Candidate updated successfully.');
    }

    public function destroy(Candidate $candidate)
    {
        $this->deleteAssets($candidate);
        $candidate->delete();

        return redirect()
            ->route('admin.candidates.index')
            ->with('status', 'Candidate deleted successfully.');
    }

    private function validateCandidate(Request $request): array
    {
        return $request->validate([
            'name_bn' => ['required', 'string', 'max:255'],
            'seat_name_bn' => ['nullable', 'string', 'max:255'],
            'symbol_name_bn' => ['nullable', 'string', 'max:255'],
            'photo' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'symbol_image' => ['nullable', 'image', 'mimes:jpg,jpeg,png,webp', 'max:2048'],
            'is_active' => ['nullable', 'boolean'],
        ]);
    }

    private function buildPayload(Request $request, array $data, ?Candidate $candidate = null): array
    {
        $payload = Arr::except($data, ['photo', 'symbol_image']);
        $payload['is_active'] = $request->boolean('is_active');

        if ($request->hasFile('photo')) {
            if ($candidate?->photo_path) {
                Storage::disk('public')->delete($candidate->photo_path);
            }
            $payload['photo_path'] = $request->file('photo')->store('candidates/photos', 'public');
        }

        if ($request->hasFile('symbol_image')) {
            if ($candidate?->symbol_path) {
                Storage::disk('public')->delete($candidate->symbol_path);
            }
            $payload['symbol_path'] = $request->file('symbol_image')->store('candidates/symbols', 'public');
        }

        return $payload;
    }

    private function deleteAssets(Candidate $candidate): void
    {
        if ($candidate->photo_path) {
            Storage::disk('public')->delete($candidate->photo_path);
        }
        if ($candidate->symbol_path) {
            Storage::disk('public')->delete($candidate->symbol_path);
        }
    }
}
